# standard imports
import datetime

# third-party imports
from sqlalchemy import Column, String, Integer, DateTime, Enum, ForeignKey, Boolean
from sqlalchemy.ext.hybrid import hybrid_method, hybrid_property
#from sqlalchemy.orm import relationship, backref
#from sqlalchemy.ext.declarative import declarative_base

# local imports
from .base import SessionBase
from ..enum import TxTypeEnum
from .otx import Otx


class Tx(SessionBase):
    __tablename__ = 'tx'

    date_added = Column(DateTime, default=datetime.datetime.utcnow)
    tx_hash = Column(String(66))
    success = Column(Boolean())


    def __init__(self, tx_hash, success):
        self.tx_hash = tx_hash
        self.success = success


class TxCache(SessionBase):
    __tablename__ = 'tx_cache'

    tx_id = Column(Integer, ForeignKey('tx.id'))
    otx_id = Column(Integer, ForeignKey('otx.id'))
    source_token_address = Column(String(42))
    destination_token_address = Column(String(42))
    sender = Column(String(42))
    recipient = Column(String(42))
    from_value = Column(Integer())
    to_value = Column(Integer())
    block_number = Column(Integer())
    tx_index = Column(Integer())
    date_updated = Column(DateTime, default=datetime.datetime.utcnow)
    date_created = Column(DateTime, default=datetime.datetime.utcnow)


    def sync(self, tx_hash):
        session = TxCache.create_session()
        tx = session.query(Tx).filter(Tx.tx_hash==tx_hash).first()
        if tx == None:
            raise FileNotFoundError('confirmed transaction record unknown {} (add a Tx first)'.format(tx_hash))
        self.tx_id = tx.id
        session.close()
        return tx


    def __init__(self, tx_hash, sender, recipient, source_token_address, destination_token_address, from_value, to_value, block_number=None, tx_index=None, outgoing=False):
        session = TxCache.create_session()
        tx = None
        if outgoing:
            tx = session.query(Otx).filter(Otx.tx_hash==tx_hash).first()
            if tx == None:
                raise FileNotFoundError('outgoing transaction record unknown {} (add a Tx first)'.format(tx_hash))
            self.otx_id = tx.id
        else:
            tx = self.sync(tx_hash)

        if tx == None:
            raise ValueError('tx hash {} (outgoing: {}) not found'.format(tx_hash, outgoing))
        session.close()

        self.sender = sender
        self.recipient = recipient
        self.source_token_address = source_token_address
        self.destination_token_address = destination_token_address
        self.from_value = from_value
        self.to_value = to_value
        self.block_number = block_number
        self.tx_index = tx_index


    @staticmethod
    def load():
        session = SessionBase.create_session()
        txc = session.query(TxCache).order_by(TxCache.block_number.desc(), TxCache.tx_index.desc()).first()
        session.close()
        return txc

#
#class TxCacheAssoc(SessionBase):
#
#    __tablename__ = 'tx_cache_assoc'
#
#    otx_id = Column(Integer, ForeignKey('otx.id'), primary_key=True),
#    tx_id = Column(Integer, ForeignKey('tx.id'), primary_key=True),
#    cache_id = Column(Integer, ForeignKey('tx_cache.id'), primary_key=True),
#    otx = relationship(Otx, backref=backref('otx_assoc'))
#
#    def __init__(self, cache_id):
#        txa = TxCacheAssoc()
#        txa.cache_id = cache_id
