# standard imports
import logging

# third-party imports
import celery
import requests
import web3

# local imports
from .rpc import RpcClient
from cic_eth.db import Otx
from cic_eth.error import PermanentTxError, TemporaryTxError
from cic_registry import ContractRegistry
from cic_eth.queue.tx import create as queue_create
from cic_eth.queue.tx import get_tx
from cic_eth.error import OutOfGasError

celery_app = celery.current_app
logg = logging.getLogger()


@celery_app.task(bind=True)
def check_gas(self, tx_hashes, txs=[], address=None, gas_required=None):
    """Check the gas level of the sender address of a transaction.

    If the account balance is not sufficient for the required gas, gas refill is requested and OutOfGasError raiser.

    If account balance is sufficient, but level of gas before spend is below "safe" threshold, gas refill is requested, and execution continues normally.

    :param tx_hashes: Transaction hashes due to be submitted
    :type tx_hashes: list of str, 0x-hex
    :param txs: Signed raw transaction data, corresponding to tx_hashes
    :type txs: list of str, 0x-hex
    :param address: Sender address
    :type address: str, 0x-hex
    :param gas_required: Gas limit * gas price for transaction, (optional, if not set will be retrived from transaction).
    :type gas_required: int
    :return: Signed raw transaction data list
    :rtype: param txs, unchanged
    """
    if len(txs) == 0:
        for i in range(len(tx_hashes)):
            o = get_tx(tx_hashes[i])
            txs.append(o['signed_tx'])
            logg.debug('ooooo {}'.format(o))
            if address == None:
                address = o['address']

    queue = self.request.delivery_info['routing_key']

    c = RpcClient(address)
    balance = c.w3.eth.getBalance(address) 
    logg.debug('address {} has gas {} needs {}'.format(address, balance, gas_required))

    if gas_required > balance:
        s_refill_gas = celery.signature(
            'cic_eth.eth.tx.refill_gas',
            [
                address,
                ],
            queue=queue,
                )
        s_refill_gas.apply_async()
        raise OutOfGasError('need to fill gas, required {}, had {}'.format(gas_required, balance))

    safe_gas = c.safe_threshold_amount()
    if balance < safe_gas:
        s_refill_gas = celery.signature(
            'cic_eth.eth.tx.refill_gas',
            [
                address,
                ],
            queue=queue,
                )
        s_refill_gas.apply_async()
        logg.debug('requested refill from {} to {}'.format(c.gas_provider(), address))

    return txs


@celery_app.task(bind=True)
def send(self, txs):
    """Send transactions to the network.

    If more than one transaction is passed to the task, it will spawn a new send task with the remaining transaction(s) after the first in the list has been processed.

    Updates the outgoing transaction queue entry to SENT on successful send.

    If a temporary error occurs, the queue entry is set to SENDFAIL.

    If a permanent error occurs due to EVM execution, queue entry value is set to REVERT.

    If a permanent error occurs due to invalid transaction data, queue entry value is set to FUBAR.

    :param txs: Signed raw transaction data
    :type txs: list of str, 0x-hex
    :raises TemporaryTxError: If unable to connect to node
    :raises PermanentTxError: If EVM execution fails immediately due to tx input, or if tx contents are invalid. 
    :return: transaction hash of sent transaction
    :rtype: str, 0x-hex
    """
    logg = celery_app.log.get_default_logger()
    if len(txs) == 0:
        raise ValueError('no transaction to send')

    queue = self.request.delivery_info['routing_key']

    tx = txs[0]
    r = None
    logg.debug('tx txt tx {}'.format(tx))
    tx_hash = web3.Web3.keccak(hexstr=tx)
    tx_hash_hex = tx_hash.hex()
    c = RpcClient()
    try:
        r = c.w3.eth.send_raw_transaction(tx)
    except requests.exceptions.ConnectionError as e:
        s_set_sent = celery.signature(
            'cic_eth.queue.tx.set_sent_status',
            [tx_hash_hex, True],
            queue=queue,
            )
        s_set_sent.apply_async()
        raise TemporaryTxError(e)
    except ValueError as e:
        s_set_final = celery.signature(
            'cic_eth.queue.tx.set_final_status',
            [tx_hash_hex, None, True],
            queue=queue,
            )
        s_set_final.apply_async()
        raise PermanentTxError(e)
    except Exception as e:
        s_set_fubar = celery.signature(
            'cic_eth.queue.tx.set_fubar',
            [tx_hash_hex],
            queue=queue,
            )
        s_set_fubar.apply_async()
        raise PermanentTxError(e)
    s_set_sent = celery.signature(
        'cic_eth.queue.tx.set_sent_status',
        [
            tx_hash_hex,
            False
            ],
            queue=queue,
        )
    s_set_sent.apply_async()

    tx_tail = txs[1:]
    if len(tx_tail) > 0:
        s = celery.signature(
            'cic_eth.eth.tx.send',
            [tx_tail],
            queue=queue,
                )
        s.apply_async()

    return r.hex()


@celery_app.task(bind=True)
def refill_gas(self, recipient_address):
    """Executes a native token transaction to fund the recipient's gas expenditures.

    :param recipient_address: Recipient in need of gas
    :type recipient_address: str, 0x-hex
    :returns: Transaction hash.
    :rtype: str, 0x-hex
    """

    queue = self.request.delivery_info['routing_key']

    c = RpcClient()
    clogg = celery_app.log.get_default_logger()
    logg.debug('refill gas from provider address {}'.format(c.gas_provider()))
    nonce = c.w3.eth.getTransactionCount(c.gas_provider(), 'pending')
    gas_price = c.gas_price()
    # defaultgas doesn't seem to work with ganache
    #gas_limit = RpcClient.w3.eth.defaultGas
    gas_limit = c.default_gas_limit
    refill_amount = c.refill_amount()
    logg.debug('gas price {} nonce {}'.format(gas_price, nonce))


    # create and sign transaction
    tx_send_gas = {
                'from': c.gas_provider(),
                'to': recipient_address,
                'gas': gas_limit,
                'gasPrice': gas_price,
                'chainId': ContractRegistry.chain_id,
                'nonce': nonce,
                'value': refill_amount,
                'data': '',
            }
    logg.debug('txsend_gas {}'.format(tx_send_gas))
    tx_send_gas_signed = c.w3.eth.sign_transaction(tx_send_gas)
    tx_hash = web3.Web3.keccak(hexstr=tx_send_gas_signed['raw'])
    tx_hash_hex = tx_hash.hex()

    q_tx = queue_create(
        nonce,
        recipient_address,
        tx_hash_hex,
        tx_send_gas_signed['raw'],   
        )
    logg.debug('added queue refill gas tx {}'.format(q_tx))

    s_send = celery.signature(
            'cic_eth.eth.tx.send',
            [[tx_send_gas_signed['raw']]],
            queue=queue,
            )
    s_send.apply_async()
    return tx_hash_hex


@celery_app.task(bind=True)
def resend_with_higher_gas(self, txold_hash_hex, gas=None, default_factor=1.1):
    session = SessionBase.create_session()
    otx = session.query(Otx).filter(Otx.tx_hash==txold_hash_hex).first()
    if otx == None:
        raise NotLocalTxError(txold_hash_hex)
    session.close()

    tx = unpack_signed_raw_tx(otx.signed_tx)

    queue = self.request.delivery_info['routing_key']

    c = RpcClient()
    if gas != None:
        tx.gasPrice = gas
    else:
        gas_price = c.gas_price()
        if tx.gasPrice > gas_price:
            logg.warning('Network gas price {} is lower than overdue tx gas price {}'.format(gas_price, tx.gasPrice))
            tx.gasPrice *= default_factor
        else:
            tx.gasPrice = gas_price

    (tx_hash_hex, tx_signed_raw_hex) = sign_and_register_tx(tx)
    s = create_check_gas_and_send_task(
            [
                [tx_hash_hex],
                [tx_signed_raw_hex],
                tx['from'],
                tx.gasPrice * tx.gas,
                ],
            queue=queue,
            )
    s.apply_async()
    return tx_hash_hex
