# standard imports
import os
import sys
import logging
import time
import argparse
import sys
import re

# third-party imports
import confini
import celery
import rlp
import web3
from web3 import HTTPProvider, WebsocketProvider

# local imports
import cic_eth
from cic_registry import ContractRegistry, bancor
from cic_eth.eth import RpcClient
from cic_eth.db import SessionBase
from cic_eth.db import OtxSync
from cic_eth.db import Otx
from cic_eth.db import TxConvertTransfer
from cic_eth.db import dsn_from_config
from cic_eth.sync import Syncer
from cic_eth.sync.error import SyncDone

logging.basicConfig(level=logging.WARNING)
logg = logging.getLogger()
logging.getLogger('websockets.protocol').setLevel(logging.CRITICAL)
logging.getLogger('web3.RequestManager').setLevel(logging.CRITICAL)
logging.getLogger('web3.providers.WebsocketProvider').setLevel(logging.CRITICAL)
logging.getLogger('web3.providers.HTTPProvider').setLevel(logging.CRITICAL)

app = celery.Celery(backend='redis://', broker='redis://')


config_dir = os.path.join('/usr/local/etc/cic-eth')

argparser = argparse.ArgumentParser(description='daemon that monitors transactions in new blocks')
argparser.add_argument('-c', type=str, default=config_dir, help='config root to use')
argparser.add_argument('--env-prefix', default=os.environ.get('CONFINI_ENV_PREFIX'), dest='env_prefix', type=str, help='environment prefix for variables to overwrite configuration')
argparser.add_argument('-q', type=str, default='cic-eth', help='celery queue to submit transaction tasks to')
argparser.add_argument('-v', help='be verbose', action='store_true')
argparser.add_argument('-vv', help='be more verbose', action='store_true')
argparser.add_argument('mode', type=str, help='sync head or backlog', default='head')
args = argparser.parse_args(sys.argv[1:])

if args.v == True:
    logging.getLogger().setLevel(logging.INFO)
elif args.vv == True:
    logging.getLogger().setLevel(logging.DEBUG)

config_dir = os.path.join(args.c)
os.makedirs(config_dir, 0o777, True)
config = confini.Config(config_dir, args.env_prefix)
config.process()
config.censor('PASSWORD', 'DATABASE')
config.censor('PASSWORD', 'SSL')
logg.debug('config loaded from {}:\n{}'.format(config_dir, config))

queue = args.q

dsn = dsn_from_config(config)
SessionBase.connect(dsn)

# TODO: move to contract registry
__convert_log_hash = '0x7154b38b5dd31bb3122436a96d4e09aba5b323ae1fd580025fab55074334c095' # sha3(Conversion(address,address,address,uint256,uint256,address)

# TODO: move to bancor package
def parse_convert_log(w3, entry):
    data = entry.data[2:]
    from_amount = int(data[:64], 16)
    to_amount = int(data[64:128], 16)
    holder_address_hex_raw = '0x' + data[-40:]
    holder_address_hex = w3.toChecksumAddress(holder_address_hex_raw)
    o = {
            'from_amount': from_amount,
            'to_amount': to_amount,
            'holder_address': holder_address_hex
            }
    logg.debug('parsed convert log {}'.format(o))
    return o


def convert_filter(w3, tx, rcpt):
    destination_token_address = None
    recipient_address = None
    amount = 0
    for l in rcpt['logs']:
        event_topic_hex = l['topics'][0].hex()
        if event_topic_hex == __convert_log_hash:
            tx_hash_hex = tx['hash'].hex()
            convert_transfer = TxConvertTransfer.get(tx_hash_hex)
            if convert_transfer.transfer_tx_hash != None:
                logg.warning('convert tx {} cache record already has transfer hash {}, skipping'.format(tx_hash_hex, convert_transfer.transfer_hash))
                continue
            recipient_address = convert_transfer.recipient_address
            logg.debug('found convert event {} recipient'.format(tx_hash_hex, recipient_address))
            r = parse_convert_log(l)
            destination_token_address = l['topics'][3][-20:]

    if destination_token_address == None:
        return None

    destination_token_address_hex = destination_token_address.hex()
    s = celery.signature(
            'cic_eth.eth.bancor.transfer_converted',
            [
                [{
                    'address': w3.toChecksumAddress(destination_token_address_hex),
                    }],
                r['holder_address'],
                recipient_address,
                r['to_amount'],
                tx_hash_hex,
                ],
                queue=queue,
            )
    logg.info('sending tx signature {}'.format(s))
    t = s.apply_async()
    logg.debug('submitted transfer after convert task uuid {} {}'.format(t, t.successful()))
    return t


def tx_filter(w3, tx, rcpt):
    tx_hash_hex = tx.hash.hex()
    otx = Otx.load(tx_hash_hex)
    if otx == None:
        logg.debug('tx {} not found locally, skipping'.format(tx_hash_hex))
        return None
    logg.info('otx found {}'.format(otx.tx_hash))
    s = celery.signature(
            'cic_eth.queue.tx.set_final_status',
            [
                tx_hash_hex,
                rcpt.blockNumber,
                rcpt.status == 0,
                ],
            )
    t = s.apply_async()
    return t



re_websocket = re.compile('^wss?://')
re_http = re.compile('^https?://')
blockchain_provider = config.get('ETH_PROVIDER')
if re.match(re_websocket, blockchain_provider) != None:
    blockchain_provider = WebsocketProvider(blockchain_provider)
elif re.match(re_http, blockchain_provider) != None:
    blockchain_provider = HTTPProvider(blockchain_provider)
else:
    raise ValueError('unknown provider url {}'.format(blockchain_provider))

def web3_constructor():
    w3 = web3.Web3(blockchain_provider)
    return (blockchain_provider, w3)
RpcClient.set_constructor(web3_constructor)



if __name__ == '__main__':
     
    c = RpcClient()

    bancor.load(c.w3, config.get('BANCOR_REGISTRY_ADDRESS'), config.get('BANCOR_DIR'))

    block_head = c.w3.eth.blockNumber
    block_sync = OtxSync.load(ContractRegistry.chain())
    block_sync.session(block_head)
    logg.info('block sync head {} backlog {} last session start {}'.format(block_sync.head(), block_sync.backlog(), block_sync.session()))

    syncer = None
    if args.mode == 'head':
        from cic_eth.sync.head import HeadSyncer
        syncer = HeadSyncer(c.w3, block_sync)
    elif args.mode == 'history':
        from cic_eth.sync.history import HistorySyncer
        syncer = HistorySyncer(block_sync)
        syncer.filter.append(tx_filter)
        syncer.filter.append(convert_filter)
    else:
        sys.stderr.write("unknown mode '{}'\n".format(args.mode))
        sys.exit(1)
   
    try:
        syncer.loop(int(config.get('SYNCER_LOOP_INTERVAL')))
    except SyncDone as e:
        sys.stderr.write("sync '{}' done at block {}\n".format(args.mode, e))

    sys.exit(0)
