# standard imports
import logging

# local imports
from cic_registry import ContractRegistry

# log output doesnt show with celery...
logg = logging.getLogger(__name__)


class TxFactory:
    """Factory class with static properties for nonce 

    BUG: This class is not thread-safe, order of nonces MUST be enforced within a single batch of txs. It is not an issue for USSD users, where there is no risk of intermingling transactions, but it may be a source of arcane buggy behavior if not secured.
    """
    inited = False
    """TxFactory.init has been successfully called"""
    nonce = 0
    """Nonce value that will be used for next transaction"""
    address = None
    """Address of spender"""
    chain_id = 0
    """Chain id of network"""
    network_contract = None
    """Bancor network contract (executes conversions on behalf of spender)"""
    gas_price = 2
    """Gas price, updated between batches"""


    def __init__(self):
        if not TxFactory.inited:
            raise RuntimeError('run TxFactory.init first')


    @staticmethod
    def init(w3, from_address):
        """Sets the spender to the provided address, and retrieves current state for the address from the network.

        :param from_address: Address of spender
        :type from_address: str, 0x-hex
        """
        if TxFactory.inited:
            logg.info('resetting txfactory')

        TxFactory.address = from_address
        TxFactory.nonce = w3.eth.getTransactionCount(from_address, 'pending') #NonceOracle.sent(w3, from_address)
        TxFactory.chain_id = ContractRegistry.chain_id
        TxFactory.gas_price = w3.eth.gasPrice 
        logg.debug('init nonce {} address {} chain_id {}'.format(TxFactory.nonce, TxFactory.address, TxFactory.chain_id))
        TxFactory.inited = True


    def next_nonce(self):
        """Returns the current cached nonce value, and increments it for next transaction.
        """
        nonce = self.nonce
        TxFactory.nonce += 1
        return nonce
