"""
.. module:: token
   :synopsis: Bancor token utility functions

.. moduleauthor:: Louis Holbrook <dev@holbrook.no>

"""
# third-party imports
import celery
import requests
import web3
import logging

# platform imports
from cic_registry import ContractRegistry
from cic_eth.db.models.tx import TxCache
from cic_eth.db.models.base import SessionBase
from cic_eth.eth import RpcClient
from cic_eth.error import TokenCountError, PermanentTxError, OutOfGasError, NotLocalTxError
from cic_eth.eth.task import sign_and_register_tx
from cic_eth.eth.task import create_check_gas_and_send_task
from cic_eth.eth.factory import TxFactory
from cic_eth.eth.util import unpack_signed_raw_tx

celery_app = celery.current_app
logg = logging.getLogger()

contract_function_signatures = {
        'transfer': 'a9059cbb',
        }


class TokenTxFactory(TxFactory):
    """Factory for creating ERC20 token transactions.
    """
    def approve(
            self,
            token_address,
            spender_address,
            amount):
        """Create an ERC20 "approve" transaction

        :param token_address: ERC20 contract address
        :type token_address: str, 0x-hex
        :param spender_address: Address to approve spending for
        :type spender_address: str, 0x-hex
        :param amount: Amount of tokens to approve
        :type amount: int
        :returns: Unsigned "approve" transaction in standard Ethereum format
        :rtype: dict
        """
        source_token = ContractRegistry.get_address(token_address)
        source_token_contract = source_token.contract
        tx_approve_buildable = source_token_contract.functions.approve(
            spender_address,
            amount,
        )
        source_token_gas = source_token.gas('transfer')

        tx_approve = tx_approve_buildable.buildTransaction({
            'from': self.address,
            'gas': source_token_gas,
            'gasPrice': self.gas_price,
            'chainId': ContractRegistry.chain_id,
            'nonce': self.next_nonce(),
            })
        return tx_approve


    def transfer(
        self,
        token_address,
        receiver_address,
        value,
        ):
        """Create an ERC20 "transfer" transaction

        :param token_address: ERC20 contract address
        :type token_address: str, 0x-hex
        :param receiver_address: Address to send tokens to
        :type receiver_address: str, 0x-hex
        :param amount: Amount of tokens to send
        :type amount: int
        :returns: Unsigned "transfer" transaction in standard Ethereum format
        :rtype: dict
        """
        source_token = ContractRegistry.get_address(token_address)
        source_token_contract = source_token.contract
        transfer_buildable = source_token_contract.functions.transfer(
                receiver_address,
                value,
                )
        source_token_gas = source_token.gas('transfer')

        tx_transfer = transfer_buildable.buildTransaction(
                {
                    'from': self.address,
                    'gas': source_token_gas,
                    'gasPrice': self.gas_price,
                    'chainId': self.chain_id,
                    'nonce': self.next_nonce(),
                })
        return tx_transfer


def unpack_transfer(data):
    f = data[2:10]
    if f != contract_function_signatures['transfer']:
        raise ValueError('Invalid transfer data ({})'.format(f))

    d = data[10:]
    return {
        'to': web3.Web3.toChecksumAddress('0x' + d[64-40:64]),
        'amount': int(d[64:], 16)
        }


@celery_app.task()
def add_token_to_registry(token_address):
    """Adds a ERC20 token to in-memory registry

    :param token_address: Address of token to add to registry
    :type token_address: str, 0x-hex
    :return: address: token address, converters: converter address
    :rtype: dict
    """
    c = RpcClient()
    t = ContractRegistry.add_token(c.w3, token_address)
    return {
        'address': t.address,
        'converters': [
            t.converter_address,
            ]
            }


@celery_app.task()
def balance(tokens, holder_address):
    """Return token balances for a list of tokens for given address

    :param tokens: Token addresses
    :type tokens: list of str, 0x-hex
    :param holder_address: Token holder address
    :type holder_address: str, 0x-hex
    :return: List of balances
    :rtype: list of int
    """
    abi = ContractRegistry.abi('ERC20Token')
    balances = []
    c = RpcClient()
    for t in tokens:
        o = c.w3.eth.contract(abi=abi, address=t['address'])
        b = o.functions.balanceOf(holder_address).call()
        logg.debug('balance {} for {}: {}'.format(t['address'], holder_address, b))
        balances.append(b)
    return b


@celery_app.task(bind=True)
def transfer(self, tokens, holder_address, receiver_address, value):
    """Transfer ERC20 tokens between addresses

    First argument is a list of tokens, to enable the task to be chained to the symbol to token address resolver function. However, it accepts only one token as argument.

    :raises TokenCountError: Either none or more then one tokens have been passed as tokens argument
    
    :param tokens: Token addresses 
    :type tokens: list of str, 0x-hex
    :param holder_address: Token holder address
    :type holder_address: str, 0x-hex
    :param receiver_address: Token receiver address
    :type receiver_address: str, 0x-hex
    :param value: Amount of token, in 'wei'
    :type value: int
    :return: Transaction hash for tranfer operation
    :rtype: str, 0x-hex
    """
    # we only allow one token, one transfer
    if len(tokens) != 1:
        raise TokenCountError

    queue = self.request.delivery_info['routing_key']

    # retrieve the token interface
    abi = ContractRegistry.abi('ERC20Token')
    t = tokens[0]

    c = RpcClient(holder_address)

    TokenTxFactory.init(c.w3, holder_address)
    txf = TokenTxFactory()

    tx_transfer = txf.transfer(t['address'], receiver_address, value)
    (tx_hash_hex, tx_signed_raw_hex) = sign_and_register_tx(tx_transfer)
    
    gas_budget = tx_transfer['gas'] * tx_transfer['gasPrice']

    s_otx_cache = celery.signature(
            'cic_eth.eth.token.otx_cache_transfer',
            [tx_hash_hex, tx_signed_raw_hex],
            queue=queue,
            )
    s_otx_cache.apply_async()

    s = create_check_gas_and_send_task(
             [tx_hash_hex],
             [tx_signed_raw_hex],
             holder_address,
             gas_budget,
             queue,
            )
    s.apply_async()
    return tx_hash_hex


@celery_app.task()
def otx_cache_transfer(
        tx_hash_hex,
        tx_signed_raw_hex,
        ):

    tx_signed_raw_bytes = bytes.fromhex(tx_signed_raw_hex[2:])
    tx = unpack_signed_raw_tx(tx_signed_raw_bytes, ContractRegistry.chain_id)
    (txc, cache_id) = cache_transfer_data(tx_hash_hex, tx, True)
    return txc


@celery_app.task()
def cache_transfer_data(
    tx_hash_hex,
    tx,
    outgoing,
        ):
    tx_data = unpack_transfer(tx['data'])
    logg.debug('tx data {}'.format(tx_data))
    logg.debug('tx {}'.format(tx))

    session = SessionBase.create_session()
    tx_cache = TxCache(
        tx_hash_hex,
        tx['from'],
        tx_data['to'],
        tx['to'],
        tx['to'],
        tx_data['amount'],
        tx_data['amount'],
        outgoing=outgoing,
            )
    session.add(tx_cache)
    session.commit()
    cache_id = tx_cache.id
    session.close()
    return (tx_hash_hex, cache_id)
