"""API for cic-eth celery tasks

.. moduleauthor:: Louis Holbrook <dev@holbrook.no>

"""
# standard imports
import logging

# third-party imports 
import celery

# local imports
from cic_registry import ContractRegistry

app = celery.current_app

logg = logging.getLogger(__file__)


class Api:
    """Creates task chains to perform well-known CIC operations.

    Each method that sends tasks returns details about the root task. The root task uuid can be provided in the callback, to enable to caller to correlate the result with individual calls. It can also be used to independently poll the completion of a task chain.

    :param callback_param: URL to HTTP POST result data, optional
    :type callback_param: str
    :param callback_task: Callback task that executes callback_param call. (Must be included by the celery worker)
    :type callback_task: function
    :param queue: Name of worker queue to submit tasks to
    :type queue: str
    """
    def __init__(self, callback_param=None, callback_task='cic_eth_callbacks.noop', queue='cic-eth'):
        self.callback_param = callback_param
        self.callback_task = callback_task
        self.queue = queue
        logg.info('api using queue {}'.format(self.queue))
        self.callback_success = None
        self.callback_error = None

        if callback_param != None:
            self.callback_success = celery.signature(
                    callback_task,
                    [
                        callback_param,
                        ],
                    queue=queue,
                    )
            self.callback_error = celery.signature(
                    callback_task,
                    [
                        callback_param,
                        1,
                        ],
                    queue=queue,
                    )       


    def convert_transfer(self, from_address, to_address, target_return, minimum_return, from_token_symbol, to_token_symbol):
        """Executes a chain of celery tasks that performs conversion between two ERC20 tokens, and transfers to a specified receipient after convert has completed.

        :param from_address: Ethereum address of sender
        :type from_address: str, 0x-hex
        :param to_address: Ethereum address of receipient
        :type to_address: str, 0x-hex
        :param target_return: Estimated return from conversion
        :type  target_return: int
        :param minimum_return: The least value of destination token return to allow
        :type minimum_return: int
        :param from_token_symbol: ERC20 token symbol of token being converted
        :type from_token_symbol: str
        :param to_token_symbol: ERC20 token symbol of token to receive
        :type to_token_symbol: str
        :returns: uuid of root task
        :rtype: celery.Task
        """
        s_tokens = celery.signature(
                'cic_eth.eth.bancor.resolve_tokens_by_symbol',
                [[from_token_symbol, to_token_symbol]],
                queue=self.queue,
                )
        s_convert = celery.signature(
                'cic_eth.eth.bancor.convert_with_default_reserve',
                [from_address, target_return, minimum_return, to_address],
                queue=self.queue,
                )
        if self.callback_param != None:
            s_convert.link(self.callback_success)
            s_tokens.link(s_convert).on_error(self.callback_error)
        else:
            s_tokens.link(s_convert)

        t = s_tokens.apply_async(queue=self.queue)
        return t


    def convert(self, from_address, target_return, minimum_return, from_token_symbol, to_token_symbol):
        """Executes a chain of celery tasks that performs conversion between two ERC20 tokens.

        :param from_address: Ethereum address of sender
        :type from_address: str, 0x-hex
        :param target_return: Estimated return from conversion
        :type  target_return: int
        :param minimum_return: The least value of destination token return to allow
        :type minimum_return: int
        :param from_token_symbol: ERC20 token symbol of token being converted
        :type from_token_symbol: str
        :param to_token_symbol: ERC20 token symbol of token to receive
        :type to_token_symbol: str
        :returns: uuid of root task
        :rtype: celery.Task
        """
        s_tokens = celery.signature(
                'cic_eth.eth.bancor.resolve_tokens_by_symbol',
                [[from_token_symbol, to_token_symbol]],
                queue=self.queue,
                )
        s_convert = celery.signature(
                'cic_eth.eth.bancor.convert_with_default_reserve',
                [from_address, target_return, minimum_return],
                queue=self.queue,
                )
        if self.callback_param != None:
            s_convert.link(self.callback_success)
            s_tokens.link(s_convert).on_error(self.callback_error)
        else:
            s_tokens.link(s_convert)

        t = s_tokens.apply_async(queue=self.queue)
        return t


    def transfer(self, from_address, to_address, value, token_symbol):
        """Executes a chain of celery tasks that performs a transfer of ERC20 tokens from one address to another.

        :param from_address: Ethereum address of sender
        :type from_address: str, 0x-hex
        :param to_address: Ethereum address of recipient
        :type to_address: str, 0x-hex
        :param value: Estimated return from conversion
        :type  value: int
        :param token_symbol: ERC20 token symbol of token to send
        :type token_symbol: str
        :returns: uuid of root task
        :rtype: celery.Task
        """
        s_tokens = celery.signature(
                'cic_eth.eth.bancor.resolve_tokens_by_symbol',
                [[token_symbol]],
                queue=self.queue,
                )
        s_transfer = celery.signature(
                'cic_eth.eth.token.transfer',
                [from_address, to_address, value],
                queue=self.queue,
                )
        if self.callback_param != None:
            s_transfer.link(self.callback_success)
            s_tokens.link(s_transfer).on_error(self.callback_error)
        else:
            s_tokens.link(s_transfer)

        t = s_tokens.apply_async(queue=self.queue)
        return t


    def balance(self, address, token_symbol):
        """Calls the provided callback with the current token balance of the given address.

        :param address: Ethereum address of holder
        :type address: str, 0x-hex
        :param token_symbol: ERC20 token symbol of token to send
        :type token_symbol: str
        :returns: uuid of root task
        :rtype: celery.Task
        """
        if self.callback_param == None:
            logg.warning('balance pointlessly called with no callback url')

        s_tokens = celery.signature(
                'cic_eth.eth.bancor.resolve_tokens_by_symbol',
                [[token_symbol]],
                queue=self.queue,
                )
        s_balance = celery.signature(
                'cic_eth.eth.token.balance',
                [address],
                queue=self.queue,
                )
        
        if self.callback_param != None:
            s_balance.link(self.callback_success)
            s_tokens.link(s_balance).on_error(self.callback_error)
        else:
            s_tokens.link(s_balance)

        t = s_tokens.apply_async(queue=self.queue)
        return t


    def create_account(self, password=''):
        """Creates a new blockchain address encrypted with the given password, and calls the provided callback with the address of the new account.

        :param password: Password to encode the password with in the backend (careful, you will have to remember it)
        :type password: str
        :returns: uuid of root task
        :rtype: celery.Task
        """
        s_account = celery.signature(
                'cic_eth.eth.account.create',
                [password],
                queue=self.queue,
                )

        if self.callback_param != None:
            s_account.link(self.callback_success)

        t = s_account.apply_async(queue=self.queue)
        return t


    def ping(self, r):
        """A noop callback ping for testing purposes.

        :returns: uuid of callback task
        :rtype: celery.Task
        """
        if self.callback_param == None:
            logg.warning('nothing to do')
            return None

        t = self.callback_success.apply_async([r])
        return t

