# standard imports
import logging
import time

# local imports
from cic_eth.eth import RpcClient

logg = logging.getLogger(__file__)


# TODO: extend blocksync model
class Syncer:
    """Base class and interface for implementing a block sync poller routine.

    :param bc_cache: Retrieves block cache cursors for chain head and latest processed block.
    :type bc_cache: object implementing methods head(height) and backlog(height)
    """
    w3 = None
    running = True

    def __init__(self, bc_cache):
        self.cursor = None
        self.bc_cache = bc_cache
        self.filter = []


    def get(self):
        """Get latest unprocessed blocks.

        :returns: list of block hash strings
        :rtype: list
        """
        raise NotImplementedError()


    def process(self, block_hash_hex):
        """Process transactions in a single block.

        :param block_hash_hex: Block hash
        :type block_hash_hex: str, 0x-hex
        """
        raise NotImplementedError()


    def loop(self, interval):
        """Loop running until the "running" property of Syncer is set to False.

        Retrieves latest unprocessed blocks and processes them.

        :param interval: Delay in seconds until next attempt if no new blocks are found.
        :type interval: int
        """
        while Syncer.running:
            c = RpcClient()
            logg.debug('loop execute')
            e = self.get()
            logg.debug('got blocks {}'.format(e))
            for block in e:
                block_number = self.process(c.w3, block.hex())
                logg.info('processed block {}'.format(block_number))
                if block_number > self.bc_cache.head():
                    self.bc_cache.head(block_number)
            time.sleep(interval)
        logg.info("Syncer no longer set to run, gracefully exiting")
