# standard imports
import logging

# third-party imports
import celery
import requests
import web3

# local imports
from .rpc import RpcClient
from cic_eth.db import Otx, SessionBase
from cic_eth.db.models.tx import TxCache
from cic_eth.error import PermanentTxError, TemporaryTxError
from cic_registry import CICRegistry
from cic_registry import zero_address
from cic_eth.queue.tx import create as queue_create
from cic_eth.queue.tx import get_tx
from cic_eth.error import OutOfGasError
from cic_eth.eth.util import unpack_signed_raw_tx
from cic_eth.eth.task import sign_and_register_tx, create_check_gas_and_send_task
from cic_eth.eth.rpc import NonceOracle
from cic_eth.error import AlreadyFillingGasError

celery_app = celery.current_app
loggg = celery_app.log.get_default_logger()
logg = logging.getLogger()


@celery_app.task(bind=True)
def check_gas(self, tx_hashes, txs=[], address=None, gas_required=None):
    """Check the gas level of the sender address of a transaction.

    If the account balance is not sufficient for the required gas, gas refill is requested and OutOfGasError raiser.

    If account balance is sufficient, but level of gas before spend is below "safe" threshold, gas refill is requested, and execution continues normally.

    :param tx_hashes: Transaction hashes due to be submitted
    :type tx_hashes: list of str, 0x-hex
    :param txs: Signed raw transaction data, corresponding to tx_hashes
    :type txs: list of str, 0x-hex
    :param address: Sender address
    :type address: str, 0x-hex
    :param gas_required: Gas limit * gas price for transaction, (optional, if not set will be retrived from transaction).
    :type gas_required: int
    :return: Signed raw transaction data list
    :rtype: param txs, unchanged
    """
    if len(txs) == 0:
        for i in range(len(tx_hashes)):
            o = get_tx(tx_hashes[i])
            txs.append(o['signed_tx'])
            logg.debug('ooooo {}'.format(o))
            if address == None:
                address = o['address']

    queue = self.request.delivery_info['routing_key']

    c = RpcClient(address)
    balance = c.w3.eth.getBalance(address) 
    logg.debug('check gas txs {}'.format(tx_hashes))
    logg.debug('address {} has gas {} needs {}'.format(address, balance, gas_required))

    logg.warning('{} {}'.format(gas_required, balance))
    if gas_required > balance:
        s_refill_gas = celery.signature(
            'cic_eth.eth.tx.refill_gas',
            [
                address,
                ],
            queue=queue,
                )
        wait_tasks = []
        s_refill_gas.apply_async()
        for tx_hash in tx_hashes:
            s = celery.signature(
                'cic_eth.queue.tx.set_waitforgas',
                [
                    tx_hash,
                    ],
                queue=queue,
                )
            wait_tasks.append(s)
        celery.group(wait_tasks)()
        raise OutOfGasError('need to fill gas, required {}, had {}'.format(gas_required, balance))

    safe_gas = c.safe_threshold_amount()
    if balance < safe_gas:
        s_refill_gas = celery.signature(
            'cic_eth.eth.tx.refill_gas',
            [
                address,
                ],
            queue=queue,
                )
        s_refill_gas.apply_async()
        logg.debug('requested refill from {} to {}'.format(c.gas_provider(), address))

    return txs


@celery_app.task(bind=True)
def hashes_to_txs(self, tx_hashes):
    #logg = celery_app.log.get_default_logger()
    if len(tx_hashes) == 0:
        raise ValueError('no transaction to send')

    queue = self.request.delivery_info['routing_key']

    #otxs = ','.format("'{}'".format(tx_hash) for tx_hash in tx_hashes)

    session = SessionBase.create_session()
    q = session.query(Otx.signed_tx)
    q = q.filter(Otx.tx_hash.in_(tx_hashes))
    tx_tuples = q.all()
    session.close()

    def __head(x):
        return x[0]

    txs = []
    for f in map(__head, tx_tuples):
        txs.append(f)

    return txs


@celery_app.task(bind=True)
def send(self, txs):
    """Send transactions to the network.

    If more than one transaction is passed to the task, it will spawn a new send task with the remaining transaction(s) after the first in the list has been processed.

    Updates the outgoing transaction queue entry to SENT on successful send.

    If a temporary error occurs, the queue entry is set to SENDFAIL.

    If a permanent error occurs due to EVM execution, queue entry value is set to REVERT.

    If a permanent error occurs due to invalid transaction data, queue entry value is set to FUBAR.

    :param txs: Signed raw transaction data
    :type txs: list of str, 0x-hex
    :raises TemporaryTxError: If unable to connect to node
    :raises PermanentTxError: If EVM execution fails immediately due to tx input, or if tx contents are invalid. 
    :return: transaction hash of sent transaction
    :rtype: str, 0x-hex
    """
    #logg = celery_app.log.get_default_logger()
    if len(txs) == 0:
        raise ValueError('no transaction to send')

    queue = self.request.delivery_info['routing_key']

    tx = txs[0]
    r = None
    logg.debug('tx txt tx {}'.format(tx))
    tx_hash = web3.Web3.keccak(hexstr=tx)
    tx_hash_hex = tx_hash.hex()
    c = RpcClient()
    try:
        r = c.w3.eth.send_raw_transaction(tx)
    except requests.exceptions.ConnectionError as e:
        s_set_sent = celery.signature(
            'cic_eth.queue.tx.set_sent_status',
            [tx_hash_hex, True],
            queue=queue,
            )
        s_set_sent.apply_async()
        raise TemporaryTxError(e)
    except ValueError as e:
        s_set_final = celery.signature(
            'cic_eth.queue.tx.set_final_status',
            [tx_hash_hex, None, True],
            queue=queue,
            )
        s_set_final.apply_async()
        raise PermanentTxError(e)
    except Exception as e:
        s_set_fubar = celery.signature(
            'cic_eth.queue.tx.set_fubar',
            [tx_hash_hex],
            queue=queue,
            )
        s_set_fubar.apply_async()
        raise PermanentTxError(e)
    s_set_sent = celery.signature(
        'cic_eth.queue.tx.set_sent_status',
        [
            tx_hash_hex,
            False
            ],
            queue=queue,
        )
    s_set_sent.apply_async()

    tx_tail = txs[1:]
    if len(tx_tail) > 0:
        s = celery.signature(
            'cic_eth.eth.tx.send',
            [tx_tail],
            queue=queue,
                )
        s.apply_async()

    return r.hex()


@celery_app.task(bind=True)
def refill_gas(self, recipient_address):
    """Executes a native token transaction to fund the recipient's gas expenditures.

    :param recipient_address: Recipient in need of gas
    :type recipient_address: str, 0x-hex
    :returns: Transaction hash.
    :rtype: str, 0x-hex
    """

    session = SessionBase.create_session()
    q = session.query(Otx.tx_hash)
    q = q.join(TxCache)
    q = q.filter(Otx.status<=0)
    q = q.filter(TxCache.from_value>0)
    c = q.count()
    logg.debug('q {} {}'.format(q, c))
    session.close()
    if c > 0:
        raise AlreadyFillingGasError(recipient_address)

    queue = self.request.delivery_info['routing_key']

    c = RpcClient()
    clogg = celery_app.log.get_default_logger()
    logg.debug('refill gas from provider address {}'.format(c.gas_provider()))
    default_nonce = c.w3.eth.getTransactionCount(c.gas_provider(), 'pending')
    nonce_generator = NonceOracle(c.gas_provider(), default_nonce)
    nonce = nonce_generator.next()
    gas_price = c.gas_price()
    # defaultgas doesn't seem to work with ganache
    #gas_limit = RpcClient.w3.eth.defaultGas
    gas_limit = c.default_gas_limit
    refill_amount = c.refill_amount()
    logg.debug('gas price {} nonce {}'.format(gas_price, nonce))


    # create and sign transaction
    tx_send_gas = {
                'from': c.gas_provider(),
                'to': recipient_address,
                'gas': gas_limit,
                'gasPrice': gas_price,
                'chainId': CICRegistry.chain_spec.chain_id(),
                'nonce': nonce,
                'value': refill_amount,
                'data': '',
            }
    logg.debug('txsend_gas {}'.format(tx_send_gas))
    tx_send_gas_signed = c.w3.eth.sign_transaction(tx_send_gas)
    tx_hash = web3.Web3.keccak(hexstr=tx_send_gas_signed['raw'])
    tx_hash_hex = tx_hash.hex()

    q_tx = queue_create(
        nonce,
        c.gas_provider(),
        #recipient_address,
        tx_hash_hex,
        tx_send_gas_signed['raw'],   
        )
    logg.debug('added queue refill gas tx {}'.format(q_tx))

    s_tx_cache = celery.signature(
            'cic_eth.eth.tx.cache_gas_refill_data',
            [
                tx_hash_hex,
                tx_send_gas,
                ],
            queue=queue,
            )
#    s_send = celery.signature(
#            'cic_eth.eth.tx.send',
#            [
#                [tx_send_gas_signed['raw']]
#                ],
#            queue=queue,
#            )
#    celery.group([s_tx_cache, s_send])()
    s_tx_cache.apply_async()
    #return tx_hash_hex
    return tx_send_gas_signed['raw']


#@celery_app.task(bind=True)
#def retry(self, sender_address, otx_status):
#    queue = self.request.delivery_info['routing_key']
#
#    chain_id = CICRegistry.chain_spec.chain_id()
#
#    session = SessionBase.create_session()
#    otxs = session.query(Otx).filter(Otx.status==otx_status).all()
#    tx_hashes_hex = []
#    tx_signed_raws_hex = []
#
#    gas = 0
#
#    for otx in otxs:
#        tx_signed_bytes = bytes.fromhex(otx.signed_tx[2:])
#        tx = unpack_signed_raw_tx(tx_signed_bytes, chain_id)
#        if tx['from'] != sender_address:
#            continue
#        tx_hashes_hex.append(otx.tx_hash)
#        tx_signed_raws_hex.append(otx.signed_tx)
#        gas += tx['gas'] * tx['gasPrice']
#
#    s = create_check_gas_and_send_task(
#        tx_signed_raws_hex,
#        sender_address,
#        gas,
#        tx_hashes_hex,
#        queue,
#        )
#    s.apply_async()
#    return tx_hashes_hex
#

@celery_app.task(bind=True)
def resend_with_higher_gas(self, txold_hash_hex, gas=None, default_factor=1.1):
    session = SessionBase.create_session()
    otx = session.query(Otx).filter(Otx.tx_hash==txold_hash_hex).first()
    if otx == None:
        raise NotLocalTxError(txold_hash_hex)
    session.close()

    tx_signed_raw_bytes = bytes.fromhex(otx.signed_tx[2:])
    tx = unpack_signed_raw_tx(tx_signed_raw_bytes, CICRegistry.chain_spec.chain_id())
    logg.debug('otx {} {}'.format(tx, otx.signed_tx))

    queue = self.request.delivery_info['routing_key']

    logg.debug('before {}'.format(tx))
    c = RpcClient()
    if gas != None:
        tx['gasPrice'] = gas
    else:
        gas_price = c.gas_price()
        if tx['gasPrice'] > gas_price:
            logg.warning('Network gas price {} is lower than overdue tx gas price {}'.format(gas_price, tx['gasPrice']))
            tx['gasPrice'] *= default_factor
        else:
            tx['gasPrice'] = gas_price
    logg.debug('after {}'.format(tx))

    (tx_hash_hex, tx_signed_raw_hex) = sign_and_register_tx(tx)
    s = create_check_gas_and_send_task(
            [tx_signed_raw_hex],
            tx['from'],
            tx['gasPrice'] * tx['gas'],
            [tx_hash_hex],
            queue=queue,
            )
    s.apply_async()
    return tx_hash_hex


@celery_app.task(bind=True)
def resume_tx(self, txpending_hash_hex):
    session = SessionBase.create_session()
    tx_signed_raw_hex = session.query(Otx.signed_tx).filter(Otx.tx_hash==txold_hash_hex).first()
    session.close()
    if otx == None:
        raise NotLocalTxError(txold_hash_hex)

    queue = self.request.delivery_info['routing_key']

    s = create_check_gas_and_send_task(
            [tx_signed_raw_hex],
            tx['from'],
            tx['gasPrice'] * tx['gas'],
            [txpending_hash_hex],
            queue=queue,
            )
    s.apply_async()
    return txpending_hash_hex


# TODO: connect to refill gas task
@celery_app.task()
def otx_cache_parse_tx(
        tx_hash_hex,
        tx_signed_raw_hex,
        ):

    tx_signed_raw_bytes = bytes.fromhex(tx_signed_raw_hex[2:])
    tx = unpack_signed_raw_tx(tx_signed_raw_bytes, CICRegistry.chain_spec.chain_id())
    (txc, cache_id) = cache_gas_refill_data(tx_hash_hex, tx)
    return txc


# TODO: connect to refill gas task
@celery_app.task()
def cache_gas_refill_data(
    tx_hash_hex,
    tx,
        ):

    tx_cache = TxCache(
        tx_hash_hex,
        tx['from'],
        tx['to'],
        zero_address,
        zero_address,
        tx['value'],
        tx['value'],
            )

    session = SessionBase.create_session()
    session.add(tx_cache)
    session.commit()
    cache_id = tx_cache.id
    session.close()
    return (tx_hash_hex, cache_id)
