"""
.. module:: tx
   :synopsis: Transaction queue tasks

.. moduleauthor:: Louis Holbrook <dev@holbrook.no>

"""
# standard imports
import logging
import time

# third-party imports
import celery
from sqlalchemy import or_
from sqlalchemy import tuple_
from sqlalchemy import func

# local imports
from cic_registry import CICRegistry
from cic_eth.db.models.otx import Otx
#from cic_eth.db.models.tx import Tx
from cic_eth.db.models.tx import TxCache
from cic_eth.db import SessionBase
from cic_eth.db.enum import StatusEnum
from cic_eth.eth.util import unpack_signed_raw_tx # TODO: should not be in same sub-path as package that imports queue.tx
from cic_eth.error import NotLocalTxError

celery_app = celery.current_app
#logg = celery_app.log.get_default_logger()
logg = logging.getLogger()


@celery_app.task()
def create(nonce, holder_address, tx_hash, signed_tx):
    """Create a new transaction queue record.

    :param nonce: Transaction nonce
    :type nonce: int
    :param holder_address: Sender address
    :type holder_address: str, 0x-hex
    :param tx_hash: Transaction hash
    :type tx_hash: str, 0x-hex
    :param signed_tx: Signed raw transaction
    :type signed_tx: 0x-hex
    :returns: transaction hash
    :rtype: str, 0x-hash
    """
    session = SessionBase.create_session()
    o = Otx(
            nonce=nonce,
            address=holder_address,
            tx_hash=tx_hash,
            signed_tx=signed_tx,
            )
    session.add(o)
    session.flush()

    # TODO: obsolete previous txs from same holder with same nonce
    q = session.query(Otx)
    q = q.join(TxCache)
    q = q.filter(Otx.nonce==nonce)
    q = q.filter(TxCache.sender==holder_address)
    q = q.filter(Otx.tx_hash!=tx_hash)

    for otx in q.all():
        logg.info('otx {} obsoleted by {}'.format(otx.tx_hash, tx_hash))
        otx.cancel(False)
        session.add(otx)
    session.commit()
    session.close()
    logg.debug('queue created nonce {} from {} hash {}'.format(nonce, holder_address, tx_hash))
    return tx_hash


# TODO: Replace set_* with single task for set status
@celery_app.task()
def set_sent_status(tx_hash, fail=False):
    """Used to set the status after a send attempt

    :param tx_hash: Transaction hash of record to modify
    :type tx_hash: str, 0x-hex
    :param fail: if True, will set a SENDFAIL status, otherwise a SENT status. (Default: False)
    :type fail: boolean
    :raises NotLocalTxError: If transaction not found in queue.
    :returns: True if tx is known, False otherwise
    :rtype: boolean
    """
    session = SessionBase.create_session()
    o = session.query(Otx).filter(Otx.tx_hash==tx_hash).first()
    if o == None:
        logg.warning('not local tx, skipping {}'.format(tx_hash))
        session.close()
        return False

    if fail:
        o.sendfail()
    else:
        o.sent()

    session.add(o)
    session.commit()
    session.close()
    return True


@celery_app.task()
def set_final_status(tx_hash, block=None, fail=False):
    """Used to set the status of an incoming transaction result. 

    :param tx_hash: Transaction hash of record to modify
    :type tx_hash: str, 0x-hex
    :param fail: if True, will set a SUCCESS status, otherwise a REVERTED status. (Default: False)
    :type fail: boolean
    :raises NotLocalTxError: If transaction not found in queue.
    """
    session = SessionBase.create_session()
    q = session.query(
            Otx.nonce.label('nonce'),
            TxCache.sender.label('sender'),
            Otx.id.label('otxid'),
            )
    q = q.join(TxCache)
    q = q.filter(Otx.tx_hash==tx_hash)
    o = q.first()

    nonce = o.nonce
    sender = o.sender
    otxid = o.otxid

    if o == None:
        session.close()
        raise NotLocalTxError('queue does not contain tx hash {}'.format(tx_hash))

    session.flush()

    q = session.query(Otx)
    q = q.filter(Otx.tx_hash==tx_hash)
    o = q.first()

    if fail:
        o.minefail(block)
    else:
        o.success(block)

    session.add(o)
    session.flush()

    q = session.query(Otx)
    q = q.join(TxCache)
    q = q.filter(Otx.nonce==nonce)
    q = q.filter(TxCache.sender==sender)
    q = q.filter(Otx.tx_hash!=tx_hash)

    for otwo in q.all():
        otwo.cancel(True)
        session.add(otwo)

    session.commit()
    session.close()


@celery_app.task()
def set_fubar(tx_hash):
    """Used to set the status when an unexpected error occurs.

    Will set the state to FUBAR

    :param tx_hash: Transaction hash of record to modify
    :type tx_hash: str, 0x-hex
    :raises NotLocalTxError: If transaction not found in queue.
    """

    session = SessionBase.create_session()
    o = session.query(Otx).filter(Otx.tx_hash==tx_hash).first()
    if o == None:
        session.close()
        raise NotLocalTxError('queue does not contain tx hash {}'.format(tx_hash))

    session.flush()

    o.fubar()
    session.add(o)
    session.commit()
    session.close()


@celery_app.task()
def set_waitforgas(tx_hash):
    """Used to set the status when a transaction must be deferred due to gas refill

    Will set the state to WAITFORGAS

    :param tx_hash: Transaction hash of record to modify
    :type tx_hash: str, 0x-hex
    :raises NotLocalTxError: If transaction not found in queue.
    """

    session = SessionBase.create_session()
    o = session.query(Otx).filter(Otx.tx_hash==tx_hash).first()
    if o == None:
        session.close()
        raise NotLocalTxError('queue does not contain tx hash {}'.format(tx_hash))

    session.flush()

    o.waitforgas()
    session.add(o)
    session.commit()
    session.close()


@celery_app.task()
def get_tx_cache(tx_hash):
    session = SessionBase.create_session()
    q = session.query(Otx)
    q = q.filter(Otx.tx_hash==tx_hash)
    otx = q.first()

    session.flush()

    q = session.query(TxCache)
    q = q.filter(TxCache.otx_id==otx.id)
    txc = q.first()

    session.close()

    tx = {
        'tx_hash': otx.tx_hash,
        'signed_tx': otx.signed_tx,
        'status': StatusEnum(otx.status).name,
        'source_token': txc.source_token_address,
        'destination_token': txc.destination_token_address,
        'sender': txc.sender,
        'recipient': txc.recipient,
        'from_value': txc.from_value,
        'to_value': txc.to_value,
        'date_created': txc.date_created,
        'date_updated': txc.date_updated,
        'date_checked': txc.date_checked,
            }

    return tx


def get_tx(tx_hash):
    """Retrieve a transaction queue record by transaction hash

    :param tx_hash: Transaction hash of record to modify
    :type tx_hash: str, 0x-hex
    :raises NotLocalTxError: If transaction not found in queue.
    :returns: nonce, address and signed_tx (raw signed transaction)
    :rtype: dict
    """
    session = SessionBase.create_session()
    tx = session.query(Otx).filter(Otx.tx_hash==tx_hash).first()
    if tx == None:
        raise NotLocalTxError('queue does not contain tx hash {}'.format(tx_hash))

    o = {
        'otx_id': tx.id,
        'nonce': tx.nonce,
        'signed_tx': tx.signed_tx,
            }
    logg.debug('get tx {}'.format(o))
    session.close()
    return o


# TODO: move query to model
def get_paused_txs(status=None, recipient=None):
    session = SessionBase.create_session()
    q = session.query(Otx)

    if status != None:
        if status == StatusEnum.PENDING or status >= StatusEnum.SENT:
            raise ValueError('not a valid paused tx value: {}'.format(status))
        q = q.filter(Otx.status==status)
        q = q.join(TxCache)
    else:
        q = q.filter(Otx.status>StatusEnum.PENDING)
        q = q.filter(Otx.status<StatusEnum.SENT)

    if recipient != None:
        q = q.filter(TxCache.recipient==recipient)

    txs = {}

    for r in q.all():
        tx_signed_bytes = bytes.fromhex(r.signed_tx[2:])
        tx = unpack_signed_raw_tx(tx_signed_bytes, int(CICRegistry.chain_spec.chain_id()))
        if recipient == None or tx['to'] == recipient:
            #gas += tx['gas'] * tx['gasPrice']
            txs[r.tx_hash] = r.signed_tx

    session.close()

    return txs


# TODO: move query to model
def get_upcoming_tx(status=StatusEnum.PENDING, recipient=None, since=None):
    session = SessionBase.create_session()
    q = session.query(
            TxCache.sender,
            func.min(Otx.nonce).label('nonce'),
            )
    q = q.join(TxCache)

    if status >= StatusEnum.SENT:
        raise ValueError('not a valid non-final tx value: {}'.format(s))
    q = q.filter(Otx.status==status)

    if recipient != None:
        q = q.filter(TxCache.recipient==recipient)

    if since != None:
        q = q.filter(TxCache.date_checked<since)

    q = q.group_by(TxCache.sender)
    logg.debug('qqqq {} {}'.format(q, since))

    txs = {}

    results = q.all()
    for r in results:
        # TODO: optimize with subquery, view, clever grouping i am not clever enough to write
        q = session.query(Otx)
        q = q.join(TxCache)
        q = q.filter(TxCache.sender==r.sender)
        q = q.filter(Otx.nonce==r.nonce)
        o = q.first()

        tx_signed_bytes = bytes.fromhex(o.signed_tx[2:])
        tx = unpack_signed_raw_tx(tx_signed_bytes, int(CICRegistry.chain_spec.chain_id()))
        #if recipient == None or tx['to'] == recipient:
            #gas += tx['gas'] * tx['gasPrice']
        txs[o.tx_hash] = o.signed_tx

    session.close()

    return txs

