# standard imports
import datetime
import logging

# third-party imports
from sqlalchemy import Column, Enum, String, Integer, DateTime, Text
from sqlalchemy.ext.hybrid import hybrid_property, hybrid_method

# local imports
from .base import SessionBase
from cic_eth.db.enum import StatusEnum
from cic_eth.db.error import TxStateChangeError
#from cic_eth.eth.util import address_hex_from_signed_tx

logg = logging.getLogger()


class Otx(SessionBase):
    __tablename__ = 'otx'

    nonce = Column(Integer)
    date_created = Column(DateTime, default=datetime.datetime.utcnow)
    tx_hash = Column(String(66))
    signed_tx = Column(Text)
    status = Column(Integer)
    block = Column(Integer)


    def set_block(self, block):
        if self.block != None:
            raise TxStateChangeError('Attempted set block {} when block was already {}'.format(block, self.block))
        self.block = block


    def waitforgas(self):
        if self.status >= StatusEnum.SENT.value:
            raise TxStateChangeError('WAITFORGAS cannot succeed final state, had {}'.format(StatusEnum(self.status).name))
        self.status = StatusEnum.WAITFORGAS


    def fubar(self):
        self.status = StatusEnum.FUBAR


    def sent(self):
        if self.status > StatusEnum.SENT:
            raise TxStateChangeError('SENT after {}'.format(StatusEnum(self.status).name))
        self.status = StatusEnum.SENT


    def sendfail(self):
        if self.status not in [StatusEnum.PENDING, StatusEnum.SENT]:
            raise TxStateChangeError('SENDFAIL must follow SENT or PENDING, but had {}'.format(StatusEnum(self.status).name))
        self.status = StatusEnum.SENDFAIL


    def minefail(self, block=None):
        if block != None:
            self.block = block
        if self.status != StatusEnum.PENDING and self.status != StatusEnum.OBSOLETED and self.status != StatusEnum.SENT:
            raise TxStateChangeError('REVERTED must follow OBSOLETED, PENDING or SENT, but had {}'.format(StatusEnum(self.status).name))
        self.status = StatusEnum.REVERTED


    def cancel(self, confirmed=False):
        status = None
        if confirmed:
            if self.status != StatusEnum.OBSOLETED:
                raise TxStateChangeError('CANCELLED must follow OBSOLETED, but had {}'.format(StatusEnum(self.status).name))
            self.status = StatusEnum.CANCELLED
        elif self.status != StatusEnum.OBSOLETED:
            if self.status > StatusEnum.SENT:
                raise TxStateChangeError('OBSOLETED must follow PENDING, SENDFAIL or SENT, but had {}'.format(StatusEnum(self.status).name))
            self.status = StatusEnum.OBSOLETED


    def success(self, block):
        if block != None:
            self.block = block
        if self.status != StatusEnum.SENT:
            raise TxStateChangeError('SUCCESS must follow SENT, but had {}'.format(StatusEnum(self.status).name))
        self.status = StatusEnum.SUCCESS
        logg.debug('success {}'.format(self.status))


    @staticmethod
    def get(status=0, limit=4096, status_exact=True):
        e = None
        session = Otx.create_session()
        if status_exact:
            e = session.query(Otx.tx_hash).filter(Otx.status==status).order_by(Otx.date_created.asc()).limit(limit).all()
        else:
            e = session.query(Otx.tx_hash).filter(Otx.status<=status).order_by(Otx.date_created.asc()).limit(limit).all()
        session.close()
        return e


    @staticmethod
    def load(tx_hash):
        session = Otx.create_session()
        q = session.query(Otx)
        q = q.filter(Otx.tx_hash==tx_hash)
        session.close()
        return q.first()


    def __init__(self, nonce, address, tx_hash, signed_tx):
        self.nonce = nonce
        self.tx_hash = tx_hash
        self.signed_tx = signed_tx
        self.status = StatusEnum.PENDING
        signed_tx_bytes = bytes.fromhex(signed_tx[2:])
       # sender_address = address_hex_from_signed_tx(signed_tx_bytes)
       # logg.debug('decoded tx {}'.format(sender_address))


class OtxSync(SessionBase):
    __tablename__ = 'otx_sync'

    blockchain = Column(String)
    height_backlog = Column(Integer)
    height_session = Column(Integer)
    height_head = Column(Integer)
    date_created = Column(DateTime, default=datetime.datetime.utcnow)
    date_updated = Column(DateTime)


    @hybrid_method
    def backlog(self, height=None):
        session = OtxSync.create_session()
        if height != None:
            self.height_backlog = height
            session.add(self)
            session.commit()
        self.date_updated = datetime.datetime.utcnow()
        height = self.height_backlog
        session.close()
        return height
    

    @hybrid_method
    def session(self, height=None):
        session = OtxSync.create_session()
        if height != None:
            self.height_session = height
            session.add(self)
            session.commit()
        self.date_updated = datetime.datetime.utcnow()
        height = self.height_session
        session.close()
        return height


    @hybrid_method
    def head(self, height=None):
        session = OtxSync.create_session()
        if height != None:
            self.height_head = height
            session.add(self)
            session.commit()
        self.date_updated = datetime.datetime.utcnow()
        height = self.height_head
        session.close()
        return height


    @hybrid_property
    def synced(self):
        return self.height_session == self.height_backlog


    @staticmethod
    def load(blockchain):
        session = SessionBase.create_session()
        s = session.query(OtxSync).filter(OtxSync.blockchain==blockchain).first()
        if s == None:
            logg.info('creating new blockchain sync record for {}'.format(blockchain))
            s = OtxSync(blockchain)
            session.add(s)
            session.commit()
        session.close()
        return s


    @staticmethod
    def latest(nonce):
        session = SessionBase.create_session()
        otx = session.query(Otx).filter(Otx.nonce==nonce).order_by(Otx.created.desc()).first()
        session.close()
        return otx


    @staticmethod
    def get_expired(datetime_threshold):
        session = SessionBase.create_session()
        q = session.query(Otx)
        q = q.filter(Otx.date_created<datetime_threshold)
        q = q.filter(Otx.status==StatusEnum.SENT)
        q = q.order_by(Otx.date_created.desc())
        q = q.group_by(Otx.nonce)
        q = q.group_by(Otx.id)
        otxs = q.all()
        session.close()
        return otxs


    def __init__(self, blockchain):
        self.blockchain = blockchain
        self.height_head = 0
        self.height_session = 0
        self.height_backlog = 0
