# standard imports
import logging

# third-party imports
import celery
import web3
from cic_registry import zero_address
from cic_registry import CICRegistry
from crypto_dev_signer.eth.web3ext import Web3 as Web3Ext
from cic_registry.error import UnknownContractError

# local imports
from cic_eth.db.models.base import SessionBase
from cic_eth.db.models.role import AccountRole
from cic_eth.db.models.otx import Otx
from cic_eth.db.models.tx import TxCache
from cic_eth.error import InitializationError
from cic_eth.eth.rpc import RpcClient

app = celery.current_app

#logg = logging.getLogger(__file__)
logg = logging.getLogger()


class AdminApi:


    def __init__(self, rpc_client, queue='cic-eth'):
        self.rpc_client = rpc_client
        self.w3 = rpc_client.w3
        self.queue = queue


    def tag_account(self, tag, address_hex):
        if not web3.Web3.isChecksumAddress(address_hex):
            raise ValueError('invalid address')
        session = SessionBase.create_session()
        role = AccountRole.set(tag, address_hex) 
        session.add(role)
        session.commit()
        session.close()


    # TODO: this is a stub, complete all checks
    def ready(self):
        addr = AccountRole.get_address('ETH_GAS_PROVIDER_ADDRESS')
        if addr == zero_address:
            raise InitializationError('missing account ETH_GAS_PROVIDER_ADDRESS')

        self.w3.eth.sign(addr, text='666f6f')


    def tx(self, tx_hash=None, tx_raw=None):

        if tx_hash != None and tx_raw != None:
            ValueError('Specify only one of hash or raw tx')

        if tx_raw != None:
            tx_hash = self.w3.keccak(hexstr=tx_raw).hex()

        s = celery.signature(
            'cic_eth.queue.tx.get_tx_cache',
            [tx_hash],
            queue=self.queue,
            )
    
        tx = s.apply_async().get()
  
        source_token = None
        destination_token = None
        try:
            source_token = CICRegistry.get_address(CICRegistry.chain_spec, tx['source_token'])
            destination_token = CICRegistry.get_address(CICRegistry.chain_spec, tx['desination_token'])
        except UnknownContractError:
            pass

        tx['sender_description'] = 'Custodial account'
        tx['recipient_description'] = 'Custodial account'

        c = RpcClient()
        if len(c.w3.eth.getCode(tx['sender'])) > 0:
            try:
                sender_contract = CICRegistry.get_address(CICRegistry.chain_spec, tx['sender'])
                tx['sender_description'] = 'Contract {}'.format(sender_contract.identifier)
            except UnknownContractError:
                tx['sender_description'] = 'Unknown contract'
        else:
            s = celery.signature(
                    'cic_eth.eth.account.have',
                    [tx['sender']],
                    queue=self.queue,
                    )
            t = s.apply_async()
            account = t.get()
            if account == None:
                tx['sender_description'] = 'Unknown account'


        if len(c.w3.eth.getCode(tx['recipient'])) > 0:
            try:
                recipient_contract = CICRegistry.get_address(CICRegistry.chain_spec, tx['recipient'])
                tx['recipient_description'] = 'Contract {}'.format(sender_contract.identifier)
            except UnknownContractError as e:
                tx['recipient_description'] = 'Unknown contract'
        else:
            s = celery.signature(
                    'cic_eth.eth.account.have',
                    [tx['recipient']],
                    queue=self.queue,
                    )
            t = s.apply_async()
            account = t.get()
            if account == None:
                tx['recipient_description'] = 'Unknown account'

        if source_token != None:
            tx['source_token_symbol'] = source_token.symbol()

        if destination_token != None:
            tx['destination_token_symbol'] = destination_token.symbol()

        tx['network_status'] = 'Not submitted'

        try:
            c.w3.eth.getTransaction(tx_hash)
            tx['network_status'] = 'Mempool'
        except web3.exceptions.TransactionNotFound:
            pass

        try:
            r = c.w3.eth.getTransactionReceipt(tx_hash)
            if r.status == 1:
                tx['network_status'] = 'Confirmed'
                tx['block'] = r.blockNumber
                tx['tx_index'] = r.transactionIndex
            else:
                tx['network_status'] = 'Reverted'
        except web3.exceptions.TransactionNotFound:
            pass

        tx['sender_gas_balance'] = c.w3.eth.getBalance(tx['sender'])
        tx['recipient_gas_balance'] = c.w3.eth.getBalance(tx['recipient'])

        return tx
