# standard imports
import logging

# third-party imports
import celery

# local impotes
from .base import Syncer
from cic_eth.queue.tx import set_final_status

app = celery.current_app
logg = logging.getLogger()


class MinedSyncer(Syncer):
    """Base implementation of block processor for mined blocks.

    Loops through all transactions, 

    :param block_hash_hex: Block hash
    :type block_hash_hex: str, 0x-hash
    """

    def __init__(self, rpc, bc_cache):
        super(MinedSyncer, self).__init__(bc_cache)
        self.rpc = rpc
        self.w3 = rpc.w3
        self.block_offset = 0
        self.tx_offset = 0


    def process(self, w3, block_hash_hex):
        b = w3.eth.getBlock(block_hash_hex)
        c = w3.eth.getBlockTransactionCount(block_hash_hex)
        s = 0
        if self.block_offset == b.number:
            s = self.tx_offset

        logg.debug('processing {} (blocknumber {}, count {}, offset {})'.format(block_hash_hex, b.number, c, s))

        for i in range(s, c):
            tx = w3.eth.getTransactionByBlock(block_hash_hex, i)
            tx_hash_hex = tx['hash'].hex()
            rcpt = w3.eth.getTransactionReceipt(tx_hash_hex)
            logg.debug('{}/{} processing tx {} from block {} {}'.format(i+1, c, tx_hash_hex, b.number, block_hash_hex))
            ours = False
            # TODO: ensure filter loop can complete on graceful shutdown
            for f in self.filter:
                #try:
                task_uuid = f(w3, tx, rcpt)
                #except Exception as e:
                #    logg.error('error in filter {} tx {}: {}'.format(f, tx_hash_hex, e))
                #    continue
                if task_uuid != None:
                    logg.debug('tx {} passed to celery task {}'.format(tx_hash_hex, task_uuid))
                    s = celery.signature(
                            'set_final_status',
                            [tx_hash_hex, rcpt['blockNumber'], not rcpt['status']],
                            )
                    s.apply_async()
                    break
            next_tx = i + 1
            if next_tx == c:
                self.bc_cache.set(b.number+1, 0)
            else:
                self.bc_cache.set(b.number, next_tx)
        if c == 0:
            logg.info('synced block {} has no transactions'.format(b.number))
            #self.bc_cache.session(b.number+1, 0)
            self.bc_cache.set(b.number+1, 0)
        return b['number']

