# standard imports
import datetime

# third-party imports
from sqlalchemy import Column, String, Integer, DateTime, Text, Boolean
from sqlalchemy.ext.hybrid import hybrid_property, hybrid_method

# local imports
from cic_eth.db.models.base import SessionBase


class BlockchainSync(SessionBase):
    __tablename__ = 'blockchain_sync'

    blockchain = Column(String)
    block_start = Column(Integer)
    tx_start = Column(Integer)
    block_cursor = Column(Integer)
    tx_cursor = Column(Integer)
    block_target = Column(Integer)
    date_created = Column(DateTime, default=datetime.datetime.utcnow)
    date_updated = Column(DateTime)


    @staticmethod
    def first(chain, session=None):
        local_session = False
        if session == None:
            session = SessionBase.create_session()
            local_session = True
        q = session.query(BlockchainSync.id)
        q = q.filter(BlockchainSync.blockchain==chain)
        o = q.first()
        if local_session:
            session.close()
        return o == None


    @staticmethod
    def get_last_live_height(current, session=None):
        local_session = False
        if session == None:
            session = SessionBase.create_session()
            local_session = True
        q = session.query(BlockchainSync)
        q = q.filter(BlockchainSync.block_target==None)
        q = q.order_by(BlockchainSync.date_created.desc())
        o = q.first()
        if local_session:
            session.close()

        return (o.block_cursor, o.tx_cursor)


    @staticmethod
    def get_unsynced(session=None):
        unsynced = []
        local_session = False
        if session == None:
            session = SessionBase.create_session()
            local_session = True
        q = session.query(BlockchainSync.id)
        q = q.filter(BlockchainSync.block_target!=None)
        q = q.filter(BlockchainSync.block_cursor<BlockchainSync.block_target)
        q = q.order_by(BlockchainSync.date_created.asc())
        for u in q.all():
            unsynced.append(u[0])
        if local_session:
            session.close()

        return unsynced


    def set(self, block_height, tx_height):
        self.block_cursor = block_height
        self.tx_cursor = tx_height


    def cursor(self):
        return (self.block_cursor, self.tx_cursor)


    def start(self):
        return (self.block_start, self.tx_start)


    def target(self):
        return self.block_target


    def chain(self):
        return self.blockchain


    def __init__(self, chain, block_start, tx_start, block_target=None):
        self.blockchain = chain
        self.block_start = block_start
        self.tx_start = tx_start
        self.block_cursor = block_start
        self.tx_cursor = tx_start
        self.block_target = block_target
        self.date_created = datetime.datetime.utcnow()
        self.date_modified = datetime.datetime.utcnow()
