# standard imports
import logging

# local imports
from cic_registry import CICRegistry
from cic_eth.eth.nonce import NonceOracle
from cic_eth.eth import RpcClient

# log output doesnt show with celery...
logg = logging.getLogger(__name__)


class TxFactory:
    """Factory class with static properties for nonce 

    BUG: This class is not thread-safe, order of nonces MUST be enforced within a single batch of txs. It is not an issue for USSD users, where there is no risk of intermingling transactions, but it may be a source of arcane buggy behavior if not secured.
    """
    inited = False
    """TxFactory.init has been successfully called"""
    chain_id = 0
    """Chain id of network TODO: deprecate, use chain_spec"""
    chain_spec = None
    """Chain spec of network"""
    network_contract = None
    """Bancor network contract (executes conversions on behalf of spender)"""
    gas_price = 100
    """Gas price, updated between batches"""

    def __init__(self, from_address):
        if not TxFactory.inited:
            raise RuntimeError('run TxFactory.init first')
        self.address = from_address

        c = RpcClient()
        self.default_nonce = c.w3.eth.getTransactionCount(from_address, 'pending')
        #TxFactory.nonce_oracle = NonceOracle(from_address, default_nonce)
        self.nonce_oracle = NonceOracle(from_address, self.default_nonce)

        logg.debug('txfactory instance default nonce {} address {} chain_id {}'.format(self.default_nonce, self.address, TxFactory.chain_id))


    @staticmethod
    def init(w3, chain_spec): #, from_address):
        """Sets the spender to the provided address, and retrieves current state for the address from the network.

        :param from_address: Address of spender
        :type from_address: str, 0x-hex
        """
        if TxFactory.inited:
            logg.info('resetting txfactory')

        TxFactory.chain_spec = chain_spec
        TxFactory.chain_id = chain_spec.chain_id()

        #TxFactory.address = from_address
        #TxFactory.gas_price = w3.eth.gasPrice 
        TxFactory.inited = True

        logg.debug('init chain_spec {}'.format(str(chain_spec)))

    def next_nonce(self):
        """Returns the current cached nonce value, and increments it for next transaction.
        """
        return self.nonce_oracle.next()
