import os
import sys
import logging
import argparse
import re
import datetime

import web3
import confini
import celery
from web3 import HTTPProvider, WebsocketProvider
from cic_registry import CICRegistry

from cic_eth.db import dsn_from_config
from cic_eth.db import SessionBase
from cic_eth.eth import RpcClient
from cic_eth.sync.retry import RetrySyncer
from cic_eth.queue.tx import get_status_tx
from cic_eth.db.enum import StatusEnum

logging.basicConfig(level=logging.WARNING)
logg = logging.getLogger()
logging.getLogger('websockets.protocol').setLevel(logging.CRITICAL)
logging.getLogger('web3.RequestManager').setLevel(logging.CRITICAL)
logging.getLogger('web3.providers.WebsocketProvider').setLevel(logging.CRITICAL)
logging.getLogger('web3.providers.HTTPProvider').setLevel(logging.CRITICAL)


config_dir = os.path.join('/usr/local/etc/cic-eth')

argparser = argparse.ArgumentParser(description='daemon that monitors transactions in new blocks')
argparser.add_argument('-c', type=str, default=config_dir, help='config root to use')
argparser.add_argument('--abi-dir', dest='abi_dir', type=str, help='Directory containing bytecode and abi')
argparser.add_argument('--env-prefix', default=os.environ.get('CONFINI_ENV_PREFIX'), dest='env_prefix', type=str, help='environment prefix for variables to overwrite configuration')
argparser.add_argument('-q', type=str, default='cic-eth', help='celery queue to submit transaction tasks to')
argparser.add_argument('-v', help='be verbose', action='store_true')
argparser.add_argument('-vv', help='be more verbose', action='store_true')
args = argparser.parse_args(sys.argv[1:])


if args.v == True:
    logging.getLogger().setLevel(logging.INFO)
elif args.vv == True:
    logging.getLogger().setLevel(logging.DEBUG)

config_dir = os.path.join(args.c)
os.makedirs(config_dir, 0o777, True)
config = confini.Config(config_dir, args.env_prefix)
config.process()
# override args
args_override = {
        'ETH_ABI_DIR': getattr(args, 'abi_dir'),
        }
config.dict_override(args_override, 'cli flag')
config.censor('PASSWORD', 'DATABASE')
config.censor('PASSWORD', 'SSL')
logg.debug('config loaded from {}:\n{}'.format(config_dir, config))

app = celery.Celery(backend=config.get('CELERY_RESULT_URL'),  broker=config.get('CELERY_BROKER_URL'))

queue = args.q

dsn = dsn_from_config(config)
SessionBase.connect(dsn)


re_websocket = re.compile('^wss?://')
re_http = re.compile('^https?://')
blockchain_provider = config.get('ETH_PROVIDER')
if re.match(re_websocket, blockchain_provider) != None:
    blockchain_provider = WebsocketProvider(blockchain_provider)
elif re.match(re_http, blockchain_provider) != None:
    blockchain_provider = HTTPProvider(blockchain_provider)
else:
    raise ValueError('unknown provider url {}'.format(blockchain_provider))

def web3_constructor():
    w3 = web3.Web3(blockchain_provider)
    return (blockchain_provider, w3)
RpcClient.set_constructor(web3_constructor)


def sendfail_filter(w3, tx, rcpt):
    logg.debug('submitting tx {} for retry'.format(tx))
    s = celery.signature(
            'cic_eth.eth.tx.resume_tx',
            [
                tx,
                CICRegistry.chain_spec.chain_id(),
                ],
            queue=queue,
            )
    s_retry_status = celery.signature(
            'cic_eth.queue.tx.set_retry',
            [],
            queue=queue,
    )
    s.link(s_retry_status)
    s.apply_async()


def dispatch():
    txs = get_status_tx(StatusEnum.RETRY, datetime.datetime.utcnow())
    if len(txs) == 0:
        logg.debug('no retry state txs found')
        return
    signed_txs = list(txs.values())
    logg.debug('signed txs {}'.format(signed_txs))
    s_send = celery.signature(
            'cic_eth.eth.tx.send',
            [signed_txs],
            queue=queue,
    )
    s_send.apply_async()


def main(): 

    c = RpcClient()

    CICRegistry.init(c.w3, config.get('CIC_REGISTRY_ADDRESS'))
    CICRegistry.add_path(config.get('ETH_ABI_DIR'))

    syncer = RetrySyncer(10, final_func=dispatch)
    syncer.filter.append(sendfail_filter)
    syncer.loop(1.0)


if __name__ == '__main__':
    main()
