# standard imports
import logging

# third-party imports
import web3
import celery
from erc20_approval_escrow import TransferApproval
from cic_registry import CICRegistry
from cic_registry.chain import ChainSpec

# local imports
from cic_eth.db.models.tx import TxCache
from cic_eth.db.models.base import SessionBase
from cic_eth.eth import RpcClient
from cic_eth.eth.factory import TxFactory
from cic_eth.eth.task import sign_and_register_tx
from cic_eth.eth.util import unpack_signed_raw_tx
from cic_eth.eth.task import create_check_gas_and_send_task
from cic_eth.error import TokenCountError

celery_app = celery.current_app
logg = logging.getLogger()

contract_function_signatures = {
        'request': 'b0addede',
        }


class TransferRequestTxFactory(TxFactory):
    """Factory for creating ERC20 token transactions.
    """
    def request(
            self,
            token_address,
            beneficiary_address,
            amount,
            chain_spec,
            ):
        transfer_approval = CICRegistry.get_contract(chain_spec, 'TransferApproval')
        fn = transfer_approval.function('createRequest')
        tx_approval_buildable = fn(beneficiary_address, token_address, amount)
        transfer_approval_gas = transfer_approval.gas('createRequest')

        tx_approval = tx_approval_buildable.buildTransaction({
            'from': self.address,
            'gas': transfer_approval_gas,
            'gasPrice': self.gas_price,
            'chainId': chain_spec.chain_id(),
            'nonce': self.next_nonce(),
            })
        return tx_approval


def unpack_transfer_approval_request(data):
    f = data[2:10]
    if f != contract_function_signatures['request']:
        raise ValueError('Invalid transfer request data ({})'.format(f))

    d = data[10:]
    return {
        'to': web3.Web3.toChecksumAddress('0x' + d[64-40:64]),
        'token': web3.Web3.toChecksumAddress('0x' + d[128-40:128]),
        'amount': int(d[128:], 16)
        }


@celery_app.task(bind=True)
def transfer_approval_request(self, tokens, holder_address, receiver_address, value, chain_str):

    if len(tokens) != 1:
        raise TokenCountError

    chain_spec = ChainSpec.from_chain_str(chain_str)

    queue = self.request.delivery_info['routing_key']

    # retrieve the token interface
    t = tokens[0]

    c = RpcClient(holder_address)

    txf = TransferRequestTxFactory(holder_address, c)

    tx_transfer = txf.request(t['address'], receiver_address, value, chain_spec)
    (tx_hash_hex, tx_signed_raw_hex) = sign_and_register_tx(tx_transfer, chain_str, None, 'cic_eth.eth.request.otx_cache_transfer_approval_request')

    gas_budget = tx_transfer['gas'] * tx_transfer['gasPrice']

    s = create_check_gas_and_send_task(
             [tx_signed_raw_hex],
             chain_str,
             holder_address,
             gas_budget,
             [tx_hash_hex],
             queue,
            )
    s.apply_async()
    return [tx_signed_raw_hex]


@celery_app.task()
def otx_cache_transfer_approval_request(
        tx_hash_hex,
        tx_signed_raw_hex,
        chain_str,
        ):

    chain_spec = ChainSpec.from_chain_str(chain_str)
    tx_signed_raw_bytes = bytes.fromhex(tx_signed_raw_hex[2:])
    tx = unpack_signed_raw_tx(tx_signed_raw_bytes, chain_spec.chain_id())
    (txc, cache_id) = cache_transfer_approval_request_data(tx_hash_hex, tx, True)
    return txc


@celery_app.task()
def cache_transfer_approval_request_data(
    tx_hash_hex,
    tx,
    outgoing,
        ):
    tx_data = unpack_transfer_approval_request(tx['data'])
    logg.debug('tx approval request data {}'.format(tx_data))
    logg.debug('tx approval request {}'.format(tx))

    session = SessionBase.create_session()
    tx_cache = TxCache(
        tx_hash_hex,
        tx['from'],
        tx_data['to'],
        tx_data['token'],
        tx_data['token'],
        tx_data['amount'],
        tx_data['amount'],
            )
    session.add(tx_cache)
    session.commit()
    cache_id = tx_cache.id
    session.close()
    return (tx_hash_hex, cache_id)
