# standard imports
import logging

# third-party imports
import celery
from cic_registry.chain import ChainSpec

# local imports
from cic_eth.eth import RpcClient
from cic_eth.queue.tx import create as queue_create
from cic_eth.error import NotLocalTxError

celery_app = celery.current_app
logg = celery_app.log.get_default_logger()


@celery_app.task()
def sign_tx(tx, chain_str):
    chain_spec = ChainSpec.from_chain_str(chain_str)
    c = RpcClient(chain_spec)
    tx_transfer_signed = c.w3.eth.sign_transaction(tx) 
    logg.debug('tx_transfer_signed {}'.format(tx_transfer_signed))
    tx_hash = c.w3.keccak(hexstr=tx_transfer_signed['raw'])
    tx_hash_hex = tx_hash.hex()
    return (tx_hash_hex, tx_transfer_signed['raw'],)


def sign_and_register_tx(tx, chain_str, queue, cache_task=None):
    """Signs the provided transaction, and adds it to the transaction queue cache (with status PENDING).

    :param tx: Standard ethereum transaction data
    :type tx: dict
    :returns: Tuple; Transaction hash, signed raw transaction data
    :rtype: tuple
    """
    (tx_hash_hex, tx_signed_raw_hex) = sign_tx(tx, chain_str)

    logg.debug('adding queue tx {}'.format(tx_hash_hex))

    s = celery.signature(
        'cic_eth.queue.tx.create',
        [
            tx['nonce'],
            tx['from'],
            tx_hash_hex,
            tx_signed_raw_hex,
            chain_str,
            ],
        queue=queue,
        )

    if cache_task != None:
        logg.debug('adding cache task {} tx {}'.format(cache_task, tx_hash_hex))
        s_cache = celery.signature(
                cache_task,
                [
                    tx_signed_raw_hex,
                    chain_str,
                    ],
                queue=queue,
                )
        s.link(s_cache)

    s.apply_async()

    return (tx_hash_hex, tx_signed_raw_hex,)


def create_check_gas_and_send_task(tx_signed_raws_hex, chain_str, holder_address, gas, tx_hashes_hex=None, queue=None):
    """Creates a celery task signature for a check_gas task linked to a send task.

    :param tx_hashes_hex: Transaction hashes
    :type tx_hashes_hex: list of str, 0x-hex
    :param tx_signed_raws_hex: Raw signed transaction data
    :type tx_signed_raws_hex: list of str, 0x-hex
    :param holder_address: Address sending the transactions
    :type holder_address: str, 0x-hex
    :param gas: Gas budget hint for transactions
    :type gas: int
    :returns: Signature of task chain
    :rtype: celery.Signature
    """

    s_check_gas = None
    if tx_hashes_hex != None:
        s_check_gas = celery.signature(
                'cic_eth.eth.tx.check_gas',
                [
                    tx_hashes_hex,
                    chain_str,
                    tx_signed_raws_hex,
                    holder_address,
                    gas,
                    ],
                queue=queue,
                )
    else:
        s_check_gas = celery.signature(
                'cic_eth.eth.tx.check_gas',
                [
                    chain_str,
                    tx_signed_raws_hex,
                    holder_address,
                    gas,
                    ],
                queue=queue,
                )
    return s_check_gas
