# standard imports
import logging
import re

# third-party imports
import celery
from celery.task.control import inspect

app = celery.current_app
logging.basicConfig(level=logging.DEBUG)
logg = logging.getLogger()

sms_r = r'^notify.sms.(\w+)'

class Api:
    """Creates task chains to perform well-known CIC operations.

    Each method that sends tasks returns details about the root task. The root task uuid can be provided in the callback, to enable to caller to correlate the result with individual calls. It can also be used to independently poll the completion of a task chain.

    :param callback_url: URL to HTTP POST result data, optional
    :type callback_url: str
    :param callback_task: Callback task that executes callback_url call. (Must be included by the celery worker)
    :type callback_task: function
    :param queue: Name of worker queue to submit tasks to
    :type queue: str
    """
    def __init__(self, queue='cic-notify'):
        i = inspect()
        task_hosts = i.registered_tasks()
        self.sms_tasks = []

        for k in task_hosts.keys():
            logg.debug('found: {} {}'.format(k, task_hosts[k]))

        for tasks in task_hosts.values():
            for task in tasks:
                if re.match(sms_r, task):
                    if task not in self.sms_tasks:
                        self.sms_tasks.append(task)
                        logg.info('added task {}'.format(task))

        self.queue = queue
        logg.info('api using queue {}'.format(self.queue))


    def sms(self, recipient, content):
        signatures = []
        for sms_task in self.sms_tasks:
            r = re.match(sms_r, sms_task)
            s = celery.signature(sms_task)
            signatures.append(s)

        g = celery.group(signatures)
        u = g.apply_async(
                args=[recipient, content],
                queue=self.queue,
                )
        return u
