"""Load Bancor contracts registry into Contract Registry cache

.. moduleauthor:: Louis Holbrook <dev@holbrook.no>

"""
# standard imports
import json
import os
import logging
import sys

# local imports
from .error import AlreadyInitializedError
from .registry import ContractRegistry, Contract, Token

logg = logging.getLogger(__file__)


contract_ids = {
    'BancorNetwork': 'BancorNetwork',
    'ConverterFactory': 'ConverterFactory',
    'BancorFormula': 'BancorFormula',
    'ConversionPathFinder': 'ConversionPathFinder',
    'ConverterRegistry': 'BancorConverterRegistry',
    'ConverterRegistryData': 'BancorConverterRegistryData',
    'ConverterBase': 'ConverterBase',
    }

def load(w3, registry_address, bancor_dir):
    """Loads and caches the state of registries from the blockchain

    :param registry_address: Registry contract address
    :type registry_address: str, 0x-hex
    :param bancor_dir: Absolute path of bancor solitidy contract repository
    :type bancor_dir: str
    """
    if ContractRegistry.loaded:
        raise AlreadyInitializedError

    contract_bundle_dir = os.path.join(bancor_dir, 'solidity', 'build', 'contracts')

    r = __load_contract_by_id(w3, 'ContractRegistry', registry_address, contract_bundle_dir)
    ContractRegistry.contracts['ContractRegistry'] = r
    ContractRegistry.addresses[r.address] = r

    for contract_id in contract_ids.keys():
        address = __get_contract_address_from_chain(w3, contract_ids[contract_id])
        c = __load_contract_by_id(w3, contract_id, address, contract_bundle_dir)
        ContractRegistry.contracts[contract_id] = c
        ContractRegistry.addresses[address] = c

    c = __load_contract_by_id(w3, 'ERC20Token', None, contract_bundle_dir)
    ContractRegistry.contracts['ERC20Token'] = c

    c = __load_contract_by_id(w3, 'LiquidTokenConverter', None, contract_bundle_dir)
    ContractRegistry.contracts['LiquidTokenConverter'] = c

    cr = ContractRegistry.contracts['ConverterRegistry'].contract
    for a in cr.functions.getAnchors().call():
        tc = __new_token_contract(w3, a)
        ContractRegistry.add_token(a, tc)

    a = r.contract.functions.getAddress('BNTToken'.encode('utf-8')).call()
    tc = __new_token_contract(w3, a)
    t = ContractRegistry.add_token(a, tc, 'RESERVE')
    ContractRegistry.contracts['BNTToken'] = t
   
    # TODO: DRY
    d_full = os.path.join(contract_bundle_dir, 'ERC20Token.json')
    f = open(d_full, 'r')
    o = json.load(f)
    f.close()
    ContractRegistry.token_abi = o['abi']

    ContractRegistry.loaded = True


def __new_token_contract(w3, address):
    erc20_abi = ContractRegistry.contracts['ERC20Token'].contract.abi
    erc20contract = w3.eth.contract(address, abi=erc20_abi)
    return erc20contract
    

def __get_contract_address_from_chain(w3, contract_registry_id):
    c = ContractRegistry.contracts['ContractRegistry'].contract
    logg.debug('looking up address for contract {}'.format(contract_registry_id))
    return c.functions.addressOf(contract_registry_id.encode('utf-8')).call()


def __load_contract_by_id(w3, contract_registry_id, address, bundle_dir):
    logg.info('loading contract {} address {} from {}'.format(contract_registry_id, address, bundle_dir))
    d_full = os.path.join(bundle_dir, contract_registry_id + '.json')
    f = open(d_full, 'r')
    o = json.load(f)
    f.close()
    contract = w3.eth.contract(address=address, abi=o['abi'], bytecode=o['bytecode'])
    return Contract(address, contract)


# TODO: add test for this call
#def add_token(w3, address):
#    contract = w3.eth.contract(address=address, abi=ContractRegistry.token_abi)
#    t = Token(address, contract, 'LIQUID')
#    logg.info('externally adding liquid token "{}" ({}) at {}'.format(t.symbol(), t.name(), address))
#    ContractRegistry.tokens[t.symbol()] = t
#    ContractRegistry.addresses[address] = t
