"""Contract Registry cache content objects

.. moduleauthor:: Louis Holbrook <dev@holbrook.no>
"""
# standard imports
import logging

# local imports
from .error import TokenExistsError

logg = logging.getLogger(__file__)



class Contract:
    """Local representation of a single contract instance in contract registry
   
    .. todo::

    :param address: Address of contract
    :type address: str, 0x-prefix
    :param address: Contract interface
    :type contract: web3.Contract object
    """
    def __init__(self, address, contract):
        self.address = address
        self.contract = contract


    def gas(self, function, data=None):
            """Retrieve cached gas limit value for given contract transaction function.

            Delegates to web3.eth.estimateGas if no cached value exists.

            :param function: Function solidity name (not function signature hash) to report gas usage for
            :type function: str
            :param data: Data that will be sent with transaction
            :type data: str, 0x-hex
            :return: Suggested gas limit
            :rtype: int
            """
            gas_max = 8000000
            logg.warning('gas budget is not implemented, replying "gas max" {} to everything'.format(8000000))
            return gas_max

    
# local representation of a single token instance registered in converter registry
class Token(Contract):
    """Local representation of a single token instance in contract registry

    :param address: Contract address of token
    :type address: str, 0x-prefix
    :param address: Token contract interface
    :type contract: web3.Contract object
    :param typ: Token type
    :type typ: str, "LIQUID" or "RESERVE"
    :param converter_address: Contract address of token converter
    :type converter_address: str, 0x-prefix, optional
    """
    valid_types = [
            'LIQUID',
            'RESERVE',
        ]

    def __init__(self, address, contract, typ='LIQUID', converter_address=None):
        if typ not in self.valid_types:
            raise ValueError('unknown token type {}'.format(typ))
        self.type = typ
        self.converter_address = converter_address
        super(Token, self).__init__(address, contract)


    def name(self):
        """Returns name of token

        :return: Name
        :rtype: str
        """
        return self.contract.functions.name().call()


    def symbol(self):
        """Returns ERC20 symbol of token

        :return: Symbol
        :rtype: str
        """
        return self.contract.functions.symbol().call()


    def decimals(self):
        """Returns number of decimals token is denominated in

        :return: Decimals
        :rtype: int
        """
        return self.contract.functions.decimals().call()



class ContractRegistry:
    """Holds the static snapshot of registry at load time.

    The contract registry should NEVER be changed.
    """
    addresses = {}
    contracts = {}
    tokens = {}
    reserves = {}
    token_abi = None
    loaded = False
    chain_type = 'eip155'
    chain_id = 1
    chain_name = None


    @staticmethod
    def chain():
        s = '{}:{}'.format(ContractRegistry.chain_type, ContractRegistry.chain_id)
        if ContractRegistry.chain_name != None:
            s += '{}:{}'.format(s, ContractRegistry.chain_name)
        return s


    @staticmethod
    def get_address(address):
        """Return contract by address

        :return: Contract if found 
        :rtype: Contract or None
        """
        return ContractRegistry.addresses.get(address)


    @staticmethod
    def cache_token_count():
        """Return number of tokens in registry cache

        :return: Number of tokens
        :rtype: int
        """
        return len(ContractRegistry.tokens)


    @staticmethod
    def get_token_by_symbol(symbol):
        """Look up token address by symbol

        :param symbol: Symbol
        :type symbol: str
        :return: Address
        :rtype: str, 0x-hex
        """
        return ContractRegistry.tokens[symbol]

    
    @staticmethod
    def get_contract(contract_name):
        """Get contract interface by Contract Registry name

        The name is an enumerated, well-known value defined by the ContractRegistryClient contract.
        :param contract_name: Name
        :type contract_name: str
        :return: Contract interface
        :rtype: web3.Contract
        """
        return ContractRegistry.contracts[contract_name]



    @staticmethod
    def get_default_reserve():
        """Retrieve reserve token address used on network

        :return: Reserve token contract address
        :rtype: str, 0x-hex
        """
        return list(ContractRegistry.reserves.values())[0]


    @staticmethod
    def add_token(address, contract, typ='LIQUID'):
        """Add token contract to local registry cache

        .. todo:: Contract should be Token object from this package

        Private function, should not be called directly

        :param address: Token address
        :type address: str, 0x-hex
        :param contract: Token contract
        :type contract: web3 contract object
        :param typ: Type of token
        :type typ: str, "LIQUID" or "RESERVE"
        """
        converter = None
        if typ == 'LIQUID':
            fn = ContractRegistry.function('ConverterRegistry', 'getConvertersByAnchors')
            converters = fn([address]).call()
            converter = converters[0]

        t = Token(address, contract, typ, converter)

        try:
            ContractRegistry.addresses[t.address]
            #ContractRegistry.get_token_by_symbol(t.symbol)
            raise TokenExistsError('{} {}'.format(t.address, t.symbol))
        except KeyError:
            pass

        logg.info('adding token "{}" ({}) at {}'.format(t.symbol(), t.name(), address))
        ContractRegistry.tokens[t.symbol()] = t
        ContractRegistry.addresses[address] = t
        if typ == 'RESERVE':
            ContractRegistry.reserves[t.symbol()] = t
        return t


    @staticmethod
    def function(contract_key, function_name):
        """Retrieve a contract function interface

        :param contract_key: Local id of contract. If the contract exists in the Contract Registry, the id will match the name used there.
        :type contract_key: str
        :param function_name: Contract function name
        :type function_name: str
        :return: Function interface
        :rtype: <web3.Contract> function
        """
        return getattr(ContractRegistry.contracts[contract_key].contract.functions, function_name)


    @staticmethod
    def abi(contract_key):
        """Return the EVM abi of given contract

        :param contract_key: Local id of contract. If the contract exists in the Contract Registry, the id will match the name used there.
        :type contract_key: str
        :return: ABI, json
        :rtype: dict
        """
        return ContractRegistry.contracts[contract_key].contract.abi
