"""Load Bancor contracts registry into Contract Registry cache

.. moduleauthor:: Louis Holbrook <dev@holbrook.no>

"""
# standard imports
import json
import os
import logging
import sys

# local imports
from .error import AlreadyInitializedError
from .error import TokenExistsError
from .registry import Contract, Token

logg = logging.getLogger(__file__)


class BancorRegistry:

    contract_ids = {
        'BancorNetwork': 'BancorNetwork',
        'ConverterFactory': 'ConverterFactory',
        'BancorFormula': 'BancorFormula',
        'ConversionPathFinder': 'ConversionPathFinder',
        'ConverterRegistry': 'BancorConverterRegistry',
        'ConverterRegistryData': 'BancorConverterRegistryData',
        'ConverterBase': 'ConverterBase',
        }


    def __init__(self, w3, chain_registry, bancor_registry_address, bancor_path):
        self.chain_registry = chain_registry
        self.address = bancor_registry_address


    def add_token(self, contract):
        """Add token contract to local registry cache

        .. todo:: Contract should be Token object from this package

        Private function, should not be called directly

        :param contract: Token contract
        :type contract: web3 contract object
        """
        converter = None
        try:
            fn = self.chain_registry.function('ConverterRegistry', 'getConvertersByAnchors')
            converters = fn([address]).call()
            converter = converters[0]
        except:
            logg.debug('token {} is not convertible'.format(contract.address))

        t = Token(contract) 

        logg.info('adding token "{}" ({}) at {}'.format(t.symbol(), t.name(), t.address()))
        self.chain_registry.add_token(contract)
        return t


    # TODO: Factor out generic erc20 token operations 
    def load(self, w3, registry_address, bancor_dir):
        """Loads and caches the state of registries from the blockchain

        :param registry_address: Registry contract address
        :type registry_address: str, 0x-hex
        :param bancor_dir: Absolute path of bancor solitidy contract repository
        :type bancor_dir: str
        """
        if self.chain_registry.loaded:
            raise AlreadyInitializedError

        self.chain_registry.chain_id = w3.eth.chainId

        contract_bundle_dir = os.path.join(bancor_dir, 'solidity', 'build', 'contracts')

        r = self.__load_contract_by_id(w3, 'ContractRegistry', registry_address, contract_bundle_dir)
        self.chain_registry.contracts['ContractRegistry'] = r
        self.chain_registry.addresses[r.address] = r

        for contract_id in BancorRegistry.contract_ids.keys():
            address = self.__get_contract_address_from_chain(w3, BancorRegistry.contract_ids[contract_id])
            c = self.__load_contract_by_id(w3, contract_id, address, contract_bundle_dir)
            self.chain_registry.contracts[contract_id] = c
            self.chain_registry.addresses[address] = c

        c = self.__load_contract_by_id(w3, 'ERC20Token', None, contract_bundle_dir)
        self.chain_registry.contracts['ERC20Token'] = c

        c = self.__load_contract_by_id(w3, 'LiquidTokenConverter', None, contract_bundle_dir)
        self.chain_registry.contracts['LiquidTokenConverter'] = c

        cr = self.chain_registry.contracts['ConverterRegistry'].contract
        for a in cr.functions.getAnchors().call():
            tc = self.__new_token_contract(w3, a)
            #self.add_token(a, tc)
            self.add_token(tc)

        a = r.contract.functions.getAddress('BNTToken'.encode('utf-8')).call()
        tc = self.__new_token_contract(w3, a)
        #t = self.add_token(a, tc, 'RESERVE')
        t = self.add_token(tc)
        self.chain_registry.contracts['BNTToken'] = t
       
        # TODO: DRY
        d_full = os.path.join(contract_bundle_dir, 'ERC20Token.json')
        f = open(d_full, 'r')
        o = json.load(f)
        f.close()
        self.chain_registry.token_abi = o['abi']

        self.chain_registry.loaded = True


    def __new_token_contract(self, w3, address):
        erc20_abi = self.chain_registry.contracts['ERC20Token'].contract.abi
        erc20contract = w3.eth.contract(address, abi=erc20_abi)
        return erc20contract
        

    def __get_contract_address_from_chain(self, w3, contract_registry_id):
        c = self.chain_registry.contracts['ContractRegistry'].contract
        logg.debug('looking up address for contract {}'.format(contract_registry_id))
        return c.functions.addressOf(contract_registry_id.encode('utf-8')).call()


    def __load_contract_by_id(self, w3, contract_registry_id, address, bundle_dir):
        logg.info('loading contract {} address {} from {}'.format(contract_registry_id, address, bundle_dir))
        d_full = os.path.join(bundle_dir, contract_registry_id + '.json')
        f = open(d_full, 'r')
        o = json.load(f)
        f.close()
        contract = w3.eth.contract(address=address, abi=o['abi'], bytecode=o['bytecode'])
        return Contract(contract)
