# standard imports
import os
import logging

# third-party imports
import web3
from web3 import Web3, HTTPProvider
import eth_tester
import pytest
import confini

# local imports
from cic_registry import ChainRegistry

bloxberg_config_sha256 = '0xf0d55c546bdcc314bd860ee9cecd46cfad0c10bf75fdcad7ab79918426f05e5f'

logg = logging.getLogger(__file__)



class EthTesterCustom(eth_tester.EthereumTester):
    w3_mock = web3.Web3(web3.Web3.EthereumTesterProvider(eth_tester.MockBackend()))

    def __init__(self, backend):
        super(EthTesterCustom, self).__init__(backend)
        logg.debug('acounts {}'.format(self.get_accounts()))
        self.accounts = {}
        self.signer = self
        for pk in self.backend.account_keys:
            pubk = pk.public_key
            address = pubk.to_checksum_address()
            logg.debug('have pk {} pubk {} addr {}'.format(pk, pk.public_key, address))
            self.accounts[address] = pk


    def get(self, address, password=None):
        pk = self.accounts['0x' + address]._raw_key
        logg.debug('get pk {} from {}'.format(pk, address))
        return pk


    def new_account(self, password=None):
        keys = KeyAPI()
        pk = os.urandom(32)
        account = self.add_account(pk.hex())
        self.accounts[account] = keys.PrivateKey(pk) 
        return account


    def sign_transaction(self, tx):
        tx['chainId'] = int(tx['chainId'])
        signer_tx = EIP155Transaction(tx, tx['nonce'], tx['chainId']) 
        tx_signed = self.signer.signTransaction(signer_tx)
        tx_signed_dict = signer_tx.serialize()
        tx_signed_dict['raw'] = '0x' + signer_tx.rlp_serialize().hex()
        return tx_signed_dict


    def send_raw_transaction(self, rlp_tx_hex):
        raw_tx = self.backend.send_raw_transaction(bytes.fromhex(rlp_tx_hex[2:]))
        return raw_tx


@pytest.fixture(scope='session')
def init_eth_tester(
        ):
    # a hack to increase the gas limit, since our contracts take more gas than what's currently hardcoded in eth-tester/py-evm
    eth_params = eth_tester.backends.pyevm.main.get_default_genesis_params({
        'gas_limit': 9000000,
        })
    eth_backend = eth_tester.PyEVMBackend(eth_params)
    eth_tester_instance = EthTesterCustom(eth_backend)
    
    return eth_tester_instance


@pytest.fixture(scope='session')
def eth_provider(
        init_eth_tester,
        ):

    return web3.Web3.EthereumTesterProvider(init_eth_tester)


@pytest.fixture(scope='session')
def w3(
        eth_provider,
        init_eth_tester,
        ):

    w3 = web3.Web3(eth_provider)
    return w3


@pytest.fixture(scope='session')
def default_chain_registry(
        default_chain_spec,
        ):
    chain_registry = ChainRegistry(default_chain_spec)
    return chain_registry
