"""Contract Registry cache content objects

.. moduleauthor:: Louis Holbrook <dev@holbrook.no>
"""
# standard imports
import os
import logging
import json

# third-party imports
import web3
import eth_accounts_index

# local imports
from .chain import ChainSpec
from .error import TokenExistsError, AlreadyInitializedError

logg = logging.getLogger(__file__)

moddir = os.path.dirname(__file__)
datadir = os.path.join(moddir, 'data')

registry_identifiers = {
        'CICRegistry': '0x{:0<64s}'.format(b'CICRegistry'.hex()),
        'BancorRegistry': '0x{:0<64s}'.format(b'BancorRegistry'.hex()),
        'AccountRegistry': '0x{:0<64s}'.format(b'AccountRegistry'.hex()),
        'TokenRegistry': '0x{:0<64s}'.format(b'TokenRegistry'.hex()),
        }

zero_address = '0x0000000000000000000000000000000000000000'


class Contract:
    """Local representation of a single contract instance in contract registry
   
    .. todo::

    :param address: Address of contract
    :type address: str, 0x-prefix
    :param address: Contract interface
    :type contract: web3.Contract object
    """
    def __init__(self, contract):
        self.contract = contract


    def abi(self):
        return self.contract.abi


    def address(self):
        return self.contract.address


    def gas(self, function, data=None):
            """Retrieve cached gas limit value for given contract transaction function.

            Delegates to web3.eth.estimateGas if no cached value exists.

            :param function: Function solidity name (not function signature hash) to report gas usage for
            :type function: str
            :param data: Data that will be sent with transaction
            :type data: str, 0x-hex
            :return: Suggested gas limit
            :rtype: int
            """
            gas_max = 8000000
            logg.warning('gas budget is not implemented, replying "gas max" {} to everything'.format(8000000))
            return gas_max


    def function(self, function_name):
        """Retrieve a contract function interface

        :param contract_key: Local id of contract. If the contract exists in the Contract Registry, the id will match the name used there.
        :type contract_key: str
        :param function_name: Contract function name
        :type function_name: str
        :return: Function interface
        :rtype: <web3.Contract> function
        """
        return getattr(self.contract.functions, function_name)


    
class Token(Contract):
    """Local representation of a single token instance in contract registry

    :param address: Contract address of token
    :type address: str, 0x-prefix
    :param contract: Token contract interface
    :type contract: web3.Contract object
    """

    def __init__(self, contract):
        super(Token, self).__init__(contract)


    def name(self):
        """Returns name of token

        :return: Name
        :rtype: str
        """
        return self.contract.functions.name().call()


    def symbol(self):
        """Returns ERC20 symbol of token

        :return: Symbol
        :rtype: str
        """
        return self.contract.functions.symbol().call()


    def decimals(self):
        """Returns number of decimals token is denominated in

        :return: Decimals
        :rtype: int
        """
        return self.contract.functions.decimals().call()


# TODO: contract_name is ambiguous after factoring out bancor registry
class ChainRegistry:
    """Caches all relevant contract objects for fast access
    """

    def __init__(self, chain_spec):
        self.addresses = {}
        self.contracts = {}
        self.tokens = {}
        self.token_abi = None
        self.loaded = False
        self.gas_provider = None
        self.chain_spec = chain_spec


    def chain(self):
        return str(self.chain_spec)


    def get_address(self, address):
        """Return contract by address

        :return: Contract if found 
        :rtype: Contract or None
        """
        return self.addresses.get(address)


    def cache_token_count(self):
        """Return number of tokens in registry cache

        :return: Number of tokens
        :rtype: int
        """
        return len(self.tokens)


    def get_token_by_symbol(self, symbol):
        """Look up token address by symbol

        :param symbol: Symbol
        :type symbol: str
        :return: Address
        :rtype: str, 0x-hex
        """
        return self.tokens[symbol]

    
    def get_contract(self, contract_name):
        """Get contract interface by Contract Registry name

        The name is an enumerated, well-known value defined by the ContractRegistryClient contract.
        :param contract_name: Name
        :type contract_name: str
        :return: Contract interface
        :rtype: web3.Contract
        """
        return self.contracts[contract_name]



    def abi(self, contract_key):
        """Return the EVM abi of given contract

        :param contract_key: Local id of contract. If the contract exists in the Contract Registry, the id will match the name used there.
        :type contract_key: str
        :return: ABI, json
        :rtype: dict
        """
        return self.contracts[contract_key].contract.abi


    def add_token(self, contract):
        if self.addresses.get(contract.address):
            raise TokenExistsError('token {} already registred'.format(contract.address))
        t = Token(contract)
        self.addresses[t.address()] = t
        self.tokens[t.symbol()] = t



class CICRegistry:

    finalized = False
    contract = None
    chain_spec = None
    bancor_chain_spec = None
    accounts_chain_spec = None
    __chains_registry = {}

    __abi = None
    __bytecode = None

    @staticmethod
    def add_chain_registry(chain_registry):
        chain_str = chain_registry.chain()
        CICRegistry.__chains_registry[chain_str] = chain_registry


#    @staticmethod
#    def set_bancor_registry(chain_spec, bancor_registry):
#        CICRegistry.__bancor_registry = bancor_registry
#        CICRegistry.__bancor_registry_chain_str = str(chain_spec)
#
#
#    @staticmethod
#    def set_accounts_registry(chain_spec, accounts_registry):
#        CICRegistry.__accounts_registry = accounts_registry
#        CICRegistry.__accounts_registry_chain_str = str(chain_spec)


    @staticmethod
    def get_contract(chain_spec, identifier):
        return CICRegistry.__chains_registry[str(chain_spec)].contracts.get(identifier)


    @staticmethod
    def get_token(chain_spec, symbol):
        return CICRegistry.__chains_registry[str(chain_spec)].get_token_by_symbol(symbol)


    @staticmethod
    def get_address(chain_spec, address):
        chain_str = str(chain_spec)
        token = CICRegistry.__chains_registry[chain_str].tokens.get(address)
        if token != None:
            return token
        return CICRegistry.__chains_registry[chain_str].contracts.get(address)
    

    @staticmethod
    def validate():
        c = CICRegistry.contract

        a = c.functions.entries(registry_identifiers['CICRegistry']).call()
        if a == zero_address:
            raise AttributeError('Registry address not set')

        a = c.functions.entries(registry_identifiers['BancorRegistry']).call()
        if a == zero_address:
            raise AttributeError('Bancor registry address not set')

        a = c.functions.entries(registry_identifiers['AccountRegistry']).call()
        if a == zero_address:
            raise AttributeError('Account registry address not set')

#        a = c.functions.entries(registry_identifiers['TokenRegistry']).call()
#        if a == zero_address:
#            raise AttributeError('Token registry address not set')



    @staticmethod
    def finalize(w3, address):
        if CICRegistry.finalized:
            raise AlreadyInitializedError('Already finalized')

        abi = CICRegistry.abi()
        CICRegistry.contract = w3.eth.contract(abi=abi, address=address)
 
        CICRegistry.validate()

        c = CICRegistry.contract

        h = c.functions.chainIdentifiers(registry_identifiers['CICRegistry']).call()
        x = web3.Web3.toText(h)
        CICRegistry.chain_spec = ChainSpec.from_chain_str(x)
        d = c.functions.chainConfigs(h).call()
        logg.warning('CICRegistry/{} chainconfig not checked: {}'.format(x, d.hex()))

        h = c.functions.chainIdentifiers(registry_identifiers['BancorRegistry']).call()
        x = web3.Web3.toText(h)
        CICRegistry.bancor_chain_spec = ChainSpec.from_chain_str(x)
        d = c.functions.chainConfigs(h).call()
        logg.warning('BancorRegistry/{} chainconfig not checked: {}'.format(x, d.hex()))

        h = c.functions.chainIdentifiers(registry_identifiers['AccountRegistry']).call()
        x = web3.Web3.toText(h)
        CICRegistry.accounts_chain_spec = ChainSpec.from_chain_str(x)
        d = c.functions.chainConfigs(h).call()
        logg.warning('AccountRegistry/{} chainconfig not checked: {}'.format(x, d.hex()))

        CICRegistry.finalized = True


    @staticmethod
    def add_token(chain_spec, contract):
        chain_str = str(chain_spec)
        CICRegistry.__chains_registry[chain_str].add_token(contract)


    @staticmethod
    def add_address(chain_spec, address):
        chain_str = str(chain_spec)
        CICRegistry.__accounts_registry.add(address)


    @staticmethod
    def tokens(chain_str):
        return CICRegistry.chains_registry[chain_str].tokens


    @staticmethod
    def abi():
        if CICRegistry.__abi == None:
            f = open(os.path.join(datadir, 'registry.abi.json'), 'r')
            CICRegistry.__abi = json.load(f)
            f.close()
        return CICRegistry.__abi


    @staticmethod
    def bytecode():
        if CICRegistry.__bytecode == None:
            f = open(os.path.join(datadir, 'registry.bin'))
            CICRegistry.__bytecode = f.read()
            f.close()
        return CICRegistry.__bytecode


