"""Contract Registry cache content objects
        logg.debug('cicregistry chain spec {}'.format(CICRegistry.chain_spec.chain_id()))

.. moduleauthor:: Louis Holbrook <dev@holbrook.no>
"""
# standard imports
import os
import logging
import json

# third-party imports
import web3
import eth_accounts_index

# local imports
from .chain import ChainSpec
from .chain import ChainRegistry
from .error import ContractExistsError
from .error import AlreadyInitializedError
from .error import UnknownContractError
from .error import UnknownChainError
from .error import ChainExistsError
from .bancor import BancorRegistry
from .contract import Contract

logg = logging.getLogger(__name__)

moddir = os.path.dirname(__file__)
datadir = os.path.join(moddir, 'data')

registry_identifiers = {
        'CICRegistry': '0x{:0<64s}'.format(b'CICRegistry'.hex()),
        'BancorRegistry': '0x{:0<64s}'.format(b'BancorRegistry'.hex()),
        'AccountRegistry': '0x{:0<64s}'.format(b'AccountRegistry'.hex()),
        'TokenRegistry': '0x{:0<64s}'.format(b'TokenRegistry'.hex()),
        }

zero_address = '0x0000000000000000000000000000000000000000'


class CICRegistry:

    finalized = False
    contract = None
    chain_spec = None
    bancor_chain_spec = None
    accounts_chain_spec = None
    __chains_registry = {}

    __abi = None
    __bytecode = None
    
#    @staticmethod
#    def clear():
#        CICRegistry.finalized = False
#        CICRegistry.contract = None
#        CICRegistry.chain_spec = None
#        CICRegistry.bancor_chain_spec = None
#        CICRegistry.accounts_chain_spec = None
#
#
    @staticmethod
    def add_chain_registry(chain_registry):
        chain_str = chain_registry.chain()
        if CICRegistry.__chains_registry.get(chain_str) != None:
            logg.debug('skip add chain registry {}'.format(chain_str))
            raise ChainExistsError(chain_str)
        CICRegistry.__chains_registry[chain_str] = chain_registry


    @staticmethod
    def get_contract(chain_spec, identifier):
        chain_str = str(chain_spec)
        return CICRegistry.__chains_registry[chain_str].contracts.get(identifier)


    @staticmethod
    def get_token(chain_spec, symbol):
        chain_str = str(chain_spec)
        return CICRegistry.__chains_registry[chain_str].get_token_by_symbol(symbol)


    @staticmethod
    def get_address(chain_spec, address):
        chain_str = str(chain_spec)
        logg.info('addresses have {}'.format(CICRegistry.__chains_registry[chain_str].addresses))
        contract = CICRegistry.__chains_registry[chain_str].addresses.get(address)
        if contract == None:
            raise UnknownContractError('{}:{}'.format(str(chain_spec), address))
        return contract
    

    @staticmethod
    def get_chain_registry(chain_spec):
        chain_str = str(chain_spec)
        return CICRegistry.__chains_registry[chain_str]


    @staticmethod
    def validate():
        c = CICRegistry.contract

        a = c.functions.entries(registry_identifiers['CICRegistry']).call()
        if a == zero_address:
            raise AttributeError('Registry address not set')

        a = c.functions.entries(registry_identifiers['BancorRegistry']).call()
        if a == zero_address:
            raise AttributeError('Bancor registry address not set')

        a = c.functions.entries(registry_identifiers['AccountRegistry']).call()
        if a == zero_address:
            raise AttributeError('Account registry address not set')

#        a = c.functions.entries(registry_identifiers['TokenRegistry']).call()
#        if a == zero_address:
#            raise AttributeError('Token registry address not set')


    @staticmethod
    def finalize(w3, address):
        if CICRegistry.finalized:
            raise AlreadyInitializedError('Already finalized')

        abi = CICRegistry.abi()
        CICRegistry.contract = w3.eth.contract(abi=abi, address=address)
 
        CICRegistry.validate()

        c = CICRegistry.contract

        h = c.functions.chainIdentifiers(registry_identifiers['CICRegistry']).call()
        x = to_text(h)
        cr = ChainRegistry(x)
        try:
            CICRegistry.add_chain_registry(cr)
        except ChainExistsError:
            pass
        CICRegistry.chain_spec = ChainSpec.from_chain_str(x)
        d = c.functions.chainConfigs(h).call()
        logg.warning('CICRegistry/{} chainconfig not checked: {}'.format(x, d.hex()))

        h = c.functions.chainIdentifiers(registry_identifiers['BancorRegistry']).call()
        x = to_text(h)
        cr = ChainRegistry(x)
        try:
            CICRegistry.add_chain_registry(cr)
        except ChainExistsError:
            pass
        CICRegistry.bancor_chain_spec = ChainSpec.from_chain_str(x)
        d = c.functions.chainConfigs(h).call()
        logg.warning('BancorRegistry/{} chainconfig not checked: {}'.format(x, d.hex()))
        abi = CICRegistry.abi('BancorRegistry')
        address = c.functions.entries(registry_identifiers['BancorRegistry']).call()
        bancor_registry_contract = w3.eth.contract(abi=abi, address=address)
        CICRegistry.__chains_registry[x].add_contract(bancor_registry_contract, 'BancorRegistry')


        h = c.functions.chainIdentifiers(registry_identifiers['AccountRegistry']).call()
        x = to_text(h)
        cr = ChainRegistry(x)
        try:
            CICRegistry.add_chain_registry(cr)
        except ChainExistsError:
            pass
        CICRegistry.accounts_chain_spec = ChainSpec.from_chain_str(x)
        d = c.functions.chainConfigs(h).call()
        logg.warning('AccountRegistry/{} chainconfig not checked: {}'.format(x, d.hex()))
        abi = CICRegistry.abi('AccountRegistry')
        address = c.functions.entries(registry_identifiers['AccountRegistry']).call()
        account_registry_contract = w3.eth.contract(abi=abi, address=address)
        CICRegistry.__chains_registry[x].add_contract(account_registry_contract, 'AccountRegistry')

        CICRegistry.finalized = True


    @staticmethod
    def add_token(chain_spec, contract):
        chain_str = str(chain_spec)
        CICRegistry.__chains_registry[chain_str].add_token(contract)


    @staticmethod
    def add_address(chain_spec, address):
        chain_str = str(chain_spec)
        CICRegistry.__accounts_registry.add(address)


    @staticmethod
    def abi(contract_identifier='CICRegistry'):
        if contract_identifier == 'CICRegistry':
            if CICRegistry.__abi == None:
                f = open(os.path.join(datadir, 'registry.abi.json'), 'r')
                CICRegistry.__abi = json.load(f)
                f.close()
            return CICRegistry.__abi
        f = open(os.path.join(datadir, '{}.abi.json'.format(contract_identifier)), 'r')
        abi = json.load(f)
        f.close()
        return abi


    @staticmethod
    def bytecode():
        if CICRegistry.__bytecode == None:
            f = open(os.path.join(datadir, 'registry.bin'))
            CICRegistry.__bytecode = f.read()
            f.close()
        return CICRegistry.__bytecode

#
#    def get_chain_registry(chain_spec, create_if_not_exist=False):
#        chain_str = str(chain_spec)
#        try:
#            registry = CICRegistry.__chains_registry[chain_spec]
#            return registry
#        except KeyError as e:
#            pass
#
#        if not create_if_not_exist:
#            raise UnknownChainError(chain_spec)
#        registry = ChainRegistry(chain_spec)
#        CICRegistry.add_chain_registry(registry)
#        return registry

#
#    @staticmethod
#    def load(w3, address, dirs):
#        if CICRegistry.finalized:
#            raise AlreadyInitializedError('Already initialized')
#
#        abi = CICRegistry.abi()
#        c = w3.eth.contract(abi=abi, address=address)
#
#        owner = c.functions.owner().call()
#        if owner != zero_address:
#            raise NotFinalizedError(address)
#
#        b = c.functions.chainIdentifiers(registry_identifiers['BancorRegistry']).call()
#        x = to_text(b)
#        bancor_chain_spec = ChainSpec.from_chain_str(x)
#        chain_registry = get_chain_registry(bancor_chain_spec, True)
#
#        address = c.functions.entries(registry_identifiers['BancorRegistry']).call()
#
#        bancor_registry = BancorRegistry(w3, chain_registry, address, dirs['bancor'])
#        bancor_registry.load()
#
#        # attempt to assign changes to CICRegistry object
#        try:
#            CICRegistry.bancor_chain_spec = bancor_chain_spec
#            CICRegistry.finalize(w3, address)
#        except:
#            CICRegistry.clear()
#            return False
#
#        return True


def to_text(b):
        b = b[:b.find(0)]
        return web3.Web3.toText(b)
