"""Creates a new CIC-Token

.. moduleauthor:: Louis Holbrook <dev@holbrook.no>
.. pgp:: 0826EDA1702D1E87C6E2875121D2E7BB88C2A746 

"""

# SPDX-License-Identifier: GPL-3.0-or-later

# standard imports
import os
import json
import argparse
import logging

# third-party imports
import web3

# local imports
from cic_registry import CICRegistry
from cic_registry import registry_identifiers
#from cic_registry.bancor import BancorDeployer
from cic_registry.bancor import BancorRegistryClient

logging.basicConfig(level=logging.WARNING)
logg = logging.getLogger()

logging.getLogger('web3').setLevel(logging.WARNING)
logging.getLogger('urllib3').setLevel(logging.WARNING)

script_dir = os.path.dirname(__file__)
data_dir = os.path.join(script_dir, '..', 'data')

argparser = argparse.ArgumentParser()
argparser.add_argument('-p', '--provider', dest='p', default='http://localhost:8545', type=str, help='Web3 provider url (http only)')
argparser.add_argument('-o', '--owner-address', dest='o', type=str, help='Owner address (must be signable by provider)')
argparser.add_argument('-r', '--registry-address', dest='r', required=True, type=str, help='CIC Registry address')
argparser.add_argument('-z', '--reserve-address', dest='z', required=True, type=str, help='Reserve token address')
argparser.add_argument('-n', '--name', dest='n', required=True, type=str, help='Token name')
argparser.add_argument('-s', '--symbol', dest='s', required=True, type=str, help='Token symbol')
argparser.add_argument('-d', '--decimals', dest='d', default=18, type=int, help='Token symbol')
argparser.add_argument('--abi-dir', dest='abi_dir', type=str, default=data_dir, help='Directory containing cic contract bytecode and abi (Default {})'.format(data_dir))
argparser.add_argument('-v', action='store_true', help='Be verbose')
argparser.add_argument('amount', type=int, help='Amount of reserve token to bind')
args = argparser.parse_args()

if args.v:
    logg.setLevel(logging.DEBUG)

def main():
    w3 = web3.Web3(web3.Web3.HTTPProvider(args.p))

    f = open(os.path.join(args.abi_dir, 'Registry.json'), 'r')
    abi = json.load(f)
    f.close()
    
    w3.eth.defaultAccount = w3.eth.accounts[0]
    if args.o != None:
        w3.eth.defaultAccount = args.o

    c = w3.eth.contract(abi=abi, address=args.r)
    bancor_registry_address = c.functions.addressOf(registry_identifiers['BancorRegistry']).call()
    logg.debug('bancor registry {}'.format(bancor_registry_address))

    #deployer = BancorDeployer(w3, args.abi_dir)
    #deployer.set_registry_contract(bancor_registry_address)

    #t = deployer.create_token(args.z, w3.eth.defaultAccount, args.n, args.s, args.d, args.amount)
    CICRegistry.set_data_dir(args.abi_dir)
    CICRegistry.finalize(w3, args.r)
    chain_registry = CICRegistry.get_chain_registry(CICRegistry.bancor_chain_spec)
    registry_client = BancorRegistryClient(w3, chain_registry, args.abi_dir)
    registry_client.load()
    registry_client.load_token(args.z)
    t = registry_client.create_token(args.z, w3.eth.defaultAccount, args.n, args.s, args.amount, decimals=args.d)
    print(t)

if __name__ == '__main__':
    main()
