"""Load Bancor contracts registry into Contract Registry cache

.. moduleauthor:: Louis Holbrook <dev@holbrook.no>

"""
# standard imports
import json
import os
import logging
import sys

# third-party imports
import web3

# local imports
from .error import AlreadyInitializedError
from .error import ContractExistsError
from .contract import Contract
from .token import Token

logg = logging.getLogger(__name__)

script_dir = os.path.dirname(os.path.realpath(__file__))
root_dir = os.path.dirname(script_dir)


logging.getLogger('web3').setLevel(logging.WARNING)
logging.getLogger('urllib3.connectionpool').setLevel(logging.WARNING)

class BancorRegistryClient:

# TODO: both chain_registry and bancor_registry_address is superfluous here, address can be retrieved from chain_registry
    # TODO: chain registry should be callback, will detach bancor for being explicit part of cic-registry package
    def __init__(self, w3, chain_registry, abi_dir):
        self.chain_registry = chain_registry
        self.abi_dir = abi_dir
        self.w3 = w3

        f = open(os.path.join(self.abi_dir, 'ERC20.json'), 'r')
        self.abi = json.load(f)
        f.close()


    def load(self, load_tokens=False):
        #f = open(os.path.join(self.abi_dir, 'Registry.json'), 'r')
        #abi = json.load(f)
        #f.close()
        #registry_contract = self.w3.eth.contract(abi=abi, address=self.address)
        #self.chain_registry.add_contract(registry_contract, 'ContractRegistry')
        registry_contract = self.chain_registry.get_contract('BancorRegistry')
        fn = registry_contract.function('addressOf')

        converter_registry_id_hex = self.w3.toHex(text=BancorRegistry.contract_ids['ConverterRegistry'])
        converter_registry_address = fn(converter_registry_id_hex).call()
        f = open(os.path.join(self.abi_dir, 'ConverterRegistry.json'), 'r')
        abi = json.load(f)
        f.close()
        converter_registry_contract = self.w3.eth.contract(abi=abi, address=converter_registry_address)
        self.chain_registry.add_contract(converter_registry_contract, 'ConverterRegistry')

        bancor_network_id_hex = self.w3.toHex(text=BancorRegistry.contract_ids['BancorNetwork'])
        bancor_network_address = fn(bancor_network_id_hex).call()
        f = open(os.path.join(self.abi_dir, 'Network.json'), 'r')
        abi = json.load(f)
        f.close()
        bancor_network_contract = self.w3.eth.contract(abi=abi, address=bancor_network_address)
        self.chain_registry.add_contract(bancor_network_contract, 'BancorNetwork')

        if load_tokens:
            for address in converter_registry_contract.functions.getConvertibleTokens().call():
                self.load_token(address)
        else:
            logg.debug('automatic token load dactivated')


    def load_token(self, address):
        abi = self.chain_registry.abi('ERC20')
        token_contract = self.w3.eth.contract(abi=abi, address=address)
        try:
            self.chain_registry.add_token(token_contract)
        except ContractExistsError as e:
            logg.warning('token address {} already registered'.format(address))

        t = Token(token_contract)
        logg.info('added token "{}" ({}) at {}'.format(t.symbol(), t.name(), t.address()))


    def create_token(self, reserve_address, holder_address, token_name, token_symbol, amount, **kwargs):
        #abi = self.abi('ERC20Token')
        #reserve_contract = self.w3.eth.contract(abi=abi, address=reserve_address)

        #network_id = web3.Web3.toHex(text='BancorNetwork')
        #network_address = BancorDeployer.registry_contract.functions.addressOf(network_id).call()

        #converterregistry_contract = self.contract('ConverterRegistry', 'BancorConverterRegistry')
        registry_contract = self.chain_registry.get_contract('BancorRegistry')
        registry_address = registry_contract.address()

        converter_registry_contract = self.chain_registry.get_contract('ConverterRegistry')
        converter_fn = converter_registry_contract.function('newConverter')

        ratio = kwargs.get('reserve_ratio', 1000000)
        max_fee = kwargs.get('max_fee', 0)
        decimals = kwargs.get('decimals', 18)

        tx_hash = converter_fn(
                0, 
                token_name,
                token_symbol,
                decimals,
                max_fee,
                [reserve_address],
                [ratio],
                ).transact({
                    'from': holder_address,
                    })

        rcpt = self.w3.eth.getTransactionReceipt(tx_hash)
        token = None
        for l in rcpt.logs:
            if l.topics[0].hex() == BancorRegistry.topics['ConverterRegistry']['ConvertibleTokenAdded']:
                convertible_token = web3.Web3.toChecksumAddress(l.topics[1][32-20:])
                if convertible_token != reserve_address:
                    token = convertible_token

        network_contract = self.chain_registry.get_contract('BancorNetwork')
        network_address = network_contract.address()
        convert_fn = network_contract.function('convert')

        reserve_contract = self.chain_registry.get_address(reserve_address)
        approve_fn = reserve_contract.function('approve')

        mint_amount = amount
        owner = holder_address
        logg.debug('owner {} is {}'.format(owner, token))

        approve_fn(network_address, 0).transact({
            'from': owner
            })
        approve_fn(network_address, mint_amount).transact({
            'from': owner
            })
        logg.debug('convert {} {} {} {}'.format(reserve_address, token, mint_amount, owner))
        convert_fn([
            reserve_address,
            token,
            token,
            ],
            mint_amount,
            mint_amount,
            ).transact({
                'from': owner,
                })
        
        return token



class BancorRegistry:

    contract_ids = {
        'BancorNetwork': 'BancorNetwork',
        'ConverterFactory': 'ConverterFactory',
        'BancorFormula': 'BancorFormula',
        'ConversionPathFinder': 'ConversionPathFinder',
        'ConverterRegistry': 'BancorConverterRegistry',
        'ConverterRegistryData': 'BancorConverterRegistryData',
        'ConverterBase': 'ConverterBase',
        }

    topics = {
            'ConverterRegistry': {
                'ConvertibleTokenAdded': '0xf2e7cf6d6ed3f77039511409a43d4fa5108f09ab71d72b014380364c910233a5',
                },
            }


    # TODO: both chain_registry and bancor_registry_address is superfluous here, address can be retrieved from chain_registry
    # TODO: chain registry should be callback, will detach bancor for being explicit part of cic-registry package
    def __init__(self, w3, chain_registry, bancor_registry_address, bancor_dir):
        self.chain_registry = chain_registry
        self.address = bancor_registry_address
        self.bancor_dir = bancor_dir
        self.w3 = w3



    def add_token(self, contract):
        """Add token contract to local registry cache

        .. todo:: Contract should be Token object from this package

        Private function, should not be called directly

        :param contract: Token contract
        :type contract: web3 contract object
        """
        converter = None
        try:
            fn = self.chain_registry.function('ConverterRegistry', 'getConvertersByAnchors')
            converters = fn([address]).call()
            converter = converters[0]
        except:
            logg.debug('token {} is not convertible'.format(contract.address))

        t = Token(contract) 

        logg.info('adding token "{}" ({}) at {}'.format(t.symbol(), t.name(), t.address()))
        self.chain_registry.add_token(contract)
        return t


    # TODO: Factor out generic erc20 token operations 
    def load(self):
        """Loads and caches the state of registries from the blockchain

        :param registry_address: Registry contract address
        :type registry_address: str, 0x-hex
        :param bancor_dir: Absolute path of bancor solitidy contract repository
        :type bancor_dir: str
        """
        if self.chain_registry.loaded:
            raise AlreadyInitializedError

        #self.chain_registry.chain_id = self.w3.eth.chainId

        #contract_bundle_dir = os.path.join(self.bancor_dir, 'solidity', 'build', 'contracts')
        contract_bundle_dir = self.bancor_dir

        logg.debug('load {}'.format(self.address)) #, self.chain_registry.chain_id))
        r = self.__load_contract_by_id('ContractRegistry', self.address, contract_bundle_dir)
        self.chain_registry.contracts['ContractRegistry'] = r
        self.chain_registry.addresses[r.address] = r

        for contract_id in BancorRegistry.contract_ids.keys():
            address = self.__get_contract_address_from_chain(BancorRegistry.contract_ids[contract_id])
            c = self.__load_contract_by_id(contract_id, address, contract_bundle_dir)
            self.chain_registry.contracts[contract_id] = c
            self.chain_registry.addresses[address] = c

        c = self.__load_contract_by_id('ERC20Token', None, contract_bundle_dir)
        self.chain_registry.contracts['ERC20Token'] = c

        c = self.__load_contract_by_id('LiquidTokenConverter', None, contract_bundle_dir)
        self.chain_registry.contracts['LiquidTokenConverter'] = c

        cr = self.chain_registry.contracts['ConverterRegistry'].contract
        for a in cr.functions.getAnchors().call():
            tc = self.__new_token_contract(a)
            self.add_token(tc)

        a = r.contract.functions.getAddress('BNTToken'.encode('utf-8')).call()
        logg.debug('bnttoken {}'.format(a))
        tc = self.__new_token_contract(a)
        t = self.add_token(tc)
        self.chain_registry.contracts['BNTToken'] = t
       
        # TODO: DRY
        d_full = os.path.join(contract_bundle_dir, 'ERC20Token.json')
        f = open(d_full, 'r')
        o = json.load(f)
        f.close()
        self.chain_registry.token_abi = o['abi']

        self.chain_registry.loaded = True


    def __new_token_contract(self, address):
        erc20_abi = self.chain_registry.contracts['ERC20Token'].contract.abi
        erc20contract = self.w3.eth.contract(address, abi=erc20_abi)
        return erc20contract
        

    def __get_contract_address_from_chain(self, contract_registry_id):
        c = self.chain_registry.contracts['ContractRegistry'].contract
        logg.debug('looking up address for contract {}'.format(contract_registry_id))
        return c.functions.addressOf(contract_registry_id.encode('utf-8')).call()


    def __load_contract_by_id(self, contract_registry_id, address, bundle_dir):
        logg.info('loading contract {} address {} from {}'.format(contract_registry_id, address, bundle_dir))
        d_full = os.path.join(bundle_dir, contract_registry_id + '.json')
        f = open(d_full, 'r')
        o = json.load(f)
        f.close()
        contract = self.w3.eth.contract(address=address, abi=o['abi'], bytecode=o['bytecode'])
        return Contract(contract, contract_registry_id)


# TODO: enable create token under new class (which does not need bancor_dir)
class BancorDeployer:

    # TODO: make instance variable
    registry_contract = None

    def __init__(self, w3, bancor_dir):
        self.w3 = w3
        #self.bancor_build_dir = os.path.join(bancor_dir, 'solidity', 'build', 'contracts')
        self.bancor_build_dir = bancor_dir
        logg.debug('bancordir {}'.format(os.path.realpath(self.bancor_build_dir)))



    def __bundle_file(self, bundle_id):
        contract_build_file = os.path.join(
                self.bancor_build_dir,
                '{}.json'.format(bundle_id),
                )
        f = open(os.path.join(contract_build_file))
        j = json.load(f)
        f.close()
        return j


    def abi(self, bundle_id):
        j = self.__bundle_file(bundle_id)
        try:
            return j['abi']
        except TypeError:
            return j



    def bytecode(self, bundle_id):
        j = self.__bundle_file(bundle_id)
        try:
            return j['bytecode']
        except TypeError:
            return j


    def contract(self, bundle_id, registry_id=None):
        if registry_id == None:
            registry_id = bundle_id
        contract_id_hex = web3.Web3.toHex(text=registry_id)
        contract_address = BancorDeployer.registry_contract.functions.addressOf(contract_id_hex).call()
        abi = self.abi(bundle_id)
        logg.debug('creating contract interface {} ({}) at address {}'.format(registry_id, bundle_id, contract_address))
        contract = self.w3.eth.contract(abi=abi, address=contract_address)
        return contract


    def set_registry_contract(self, registry_address):
        registry_build_file = os.path.join(self.bancor_build_dir, 'ContractRegistry.json')
        f = open(os.path.join(registry_build_file))
        j = json.load(f)
        f.close()
        try:
            BancorDeployer.registry_contract = self.w3.eth.contract(abi=j['abi'], address=registry_address)
        except TypeError:
            BancorDeployer.registry_contract = self.w3.eth.contract(abi=j, address=registry_address)


    # TODO: DRY
    def deploy(self):
        bancor_build_dir = self.bancor_build_dir

        # deploy registry
        registry_build_file = os.path.join(bancor_build_dir, 'ContractRegistry.json')
        f = open(os.path.join(registry_build_file))
        j = json.load(f)
        f.close()
        registry_constructor = self.w3.eth.contract(abi=j['abi'], bytecode=j['bytecode'])
        tx = registry_constructor.constructor().transact()
        rcpt = self.w3.eth.getTransactionReceipt(tx)
        registry_address = rcpt['contractAddress']
        self.set_registry_contract(registry_address)
        #BancorDeployer.registry_contract = self.w3.eth.contract(abi=j['abi'], address=registry_address)

        # deploy converter factory contract for creating liquid token exchanges
        build_file = os.path.join(bancor_build_dir, 'LiquidTokenConverterFactory.json')
        f = open(build_file)
        j = json.load(f)
        f.close()
        converterfactory_constructor = self.w3.eth.contract(abi=j['abi'], bytecode=j['bytecode'])
        tx = converterfactory_constructor.constructor().transact()
        rcpt = self.w3.eth.getTransactionReceipt(tx)
        converter_factory_address = rcpt['contractAddress']

        # deploy the remaining contracts managed by the registry
        for k in BancorRegistry.contract_ids.keys():
            build_file = os.path.join(bancor_build_dir, '{}.json'.format(k))
            f = open(build_file)
            j = json.load(f)
            f.close()
            contract_constructor = self.w3.eth.contract(abi=j['abi'], bytecode=j['bytecode'])
            tx = None

            # include the registry address as constructor parameters for the contracts that require it
            if k in ['ConverterRegistry', 'ConverterRegistryData', 'BancorNetwork', 'ConversionPathFinder']:
                tx = contract_constructor.constructor(registry_address).transact()
            else:
                tx = contract_constructor.constructor().transact()
            rcpt = self.w3.eth.getTransactionReceipt(tx)
            contract_address = rcpt['contractAddress']

            # register contract in registry
            key_hex = self.w3.toHex(text=BancorRegistry.contract_ids[k])
            BancorDeployer.registry_contract.functions.registerAddress(key_hex, contract_address).transact()
            contract = self.w3.eth.contract(abi=j['abi'], address=contract_address)

            # bancor formula needs to be initialized before use
            if k == 'BancorFormula':
                logg.debug('init bancor formula {}'.format(contract_address))
                contract.functions.init().transact()

            # converter factory needs liquid token converter factory to be able to issue our liquid tokens
            if k == 'ConverterFactory':
                logg.debug('register converter factory {}'.format(converter_factory_address))
                contract.functions.registerTypedConverterFactory(converter_factory_address).transact()
        
        logg.info('deployed registry at address {}'.format(registry_address))
        return BancorDeployer.registry_contract.address


    def set_default_reserve(self, default_reserve_address):
        # register reserve token as bancor hub token
        key_hex = web3.Web3.toHex(text='BNTToken')
        BancorDeployer.registry_contract.functions.registerAddress(key_hex, default_reserve_address).transact()


    def create_token(self, reserve_address, holder_address, name, symbol, decimals, amount, **kwargs):
        abi = self.abi('ERC20Token')
        reserve_contract = self.w3.eth.contract(abi=abi, address=reserve_address)

        network_id = web3.Web3.toHex(text='BancorNetwork')
        network_address = BancorDeployer.registry_contract.functions.addressOf(network_id).call()

        converterregistry_contract = self.contract('ConverterRegistry', 'BancorConverterRegistry')

        tx_hash = self.__create_converter(converterregistry_contract, reserve_address, holder_address, name, symbol, **kwargs)
        rcpt = self.w3.eth.getTransactionReceipt(tx_hash)
        token = None
        for l in rcpt.logs:
            if l.topics[0].hex() == BancorRegistry.topics['ConverterRegistry']['ConvertibleTokenAdded']:
                convertible_token = web3.Web3.toChecksumAddress(l.topics[1][32-20:])
                if convertible_token != reserve_address:
                    token = convertible_token

        network_contract = self.contract('BancorNetwork')

        mint_amount = amount
        owner = holder_address
        logg.debug('owner {} is {}'.format(owner, token))
        reserve_contract.functions.approve(network_address, 0).transact({
            'from': owner
            })
        reserve_contract.functions.approve(network_address, mint_amount).transact({
            'from': owner
            })
        logg.debug('convert {} {} {} {}'.format(reserve_address, token, mint_amount, owner))
        network_contract.functions.convert([
            reserve_address,
            token,
            token,
            ],
            mint_amount,
            mint_amount,
            ).transact({
                'from': owner,
                })
        
        return token

    def __create_converter(self, converterregistry_contract, reserve_address, owner_address, token_name, token_symbol, **kwargs):
        return converterregistry_contract.functions.newConverter(
                0, 
                token_name,
                token_symbol,
                18,
                100000,
                [reserve_address],
                [250000],
                ).transact({
                    'from': owner_address,
                    })


