# standard imports
import os
import json

# local imports
from .error import ContractExistsError
from .token import Token
from .contract import Contract

script_dir = os.path.dirname(os.path.realpath(__file__))
data_dir = os.path.join(script_dir, 'data')

class ChainSpec:

    def __init__(self, chain_config, block_offset=0):
        self.chain_config = chain_config
        self.block_offset = block_offset # earliest block to consider
    

    def chain_id(self):
        return self.chain_config['params']['networkID']


    def chain_name(self):
        return self.chain_config['name']


    def __str__(self):
        return '{}:{}'.format(self.chain_name(), self.chain_id())


    @staticmethod
    def from_chain_str(s):
        (chain_name, chain_id) = s.split(':')
        return ChainSpec({
            'params': {
                'networkID': int(chain_id),
                },
            'name': chain_name,
            }
            )


# TODO: contract_name is ambiguous after factoring out bancor registry
class ChainRegistry:
    """Caches all relevant contract objects for fast access
    """

    def __init__(self, chain_spec):
        self.addresses = {}
        self.contracts = {}
        self.tokens = {}
        self.erc20_abi = None
        self.loaded = False
        self.gas_provider = None
        self.chain_spec = chain_spec

        f = open(os.path.join(data_dir, 'ERC20.json'), 'r')
        abi = json.load(f)
        f.close()
        self.erc20_abi = abi


    def chain(self):
        return str(self.chain_spec)


    def get_address(self, address):
        """Return contract by address

        :return: Contract if found 
        :rtype: Contract or None
        """
        return self.addresses.get(address)


    def cache_token_count(self):
        """Return number of tokens in registry cache

        :return: Number of tokens
        :rtype: int
        """
        return len(self.tokens)


    def get_token_by_symbol(self, symbol):
        """Look up token address by symbol

        :param symbol: Symbol
        :type symbol: str
        :return: Address
        :rtype: str, 0x-hex
        """
        return self.tokens[symbol]
        
    
    def get_contract(self, contract_name):
        """Get contract interface by Contract Registry name

        The name is an enumerated, well-known value defined by the ContractRegistryClient contract.
        :param contract_name: Name
        :type contract_name: str
        :return: Contract interface
        :rtype: web3.Contract
        """
        return self.contracts[contract_name]


    def abi(self, contract_key):
        """Return the EVM abi of given contract

        :param contract_key: Local id of contract. If the contract exists in the Contract Registry, the id will match the name used there.
        :type contract_key: str
        :return: ABI, json
        :rtype: dict
        """
        if contract_key == 'ERC20':
            return self.erc20_abi

        return self.contracts[contract_key].contract.abi


    def add_token(self, contract):
        if self.addresses.get(contract.address):
            raise ContractExistsError('token {} already registred'.format(contract.address))
        t = Token(contract)
        self.addresses[t.address()] = t
        self.tokens[t.symbol()] = t
        return t


    def add_contract(self, contract, identifier):
        if self.addresses.get(contract.address):
            raise ContractExistsError('token {} already registred'.format(contract.address))
        c = Contract(contract, identifier)
        self.addresses[c.address()] = c
        self.contracts[identifier] = c
        return c


