"""Contract Registry cache content objects
        logg.debug('cicregistry chain spec {}'.format(CICRegistry.chain_spec.chain_id()))

.. moduleauthor:: Louis Holbrook <dev@holbrook.no>
"""
# standard imports
import os
import logging
import json

# third-party imports
import web3
import eth_accounts_index

# local imports
from .chain import ChainSpec
from .chain import ChainRegistry
from .error import ContractExistsError
from .error import AlreadyInitializedError
from .error import UnknownContractError
from .error import UnknownChainError
from .error import ChainExistsError
from .bancor import BancorRegistry
from .contract import Contract

logg = logging.getLogger(__name__)

moddir = os.path.dirname(__file__)
datadir = os.path.join(moddir, 'data')

registry_identifiers = {
        'CICRegistry': '0x{:0<64s}'.format(b'CICRegistry'.hex()),
        'BancorRegistry': '0x{:0<64s}'.format(b'BancorRegistry'.hex()),
        'AccountRegistry': '0x{:0<64s}'.format(b'AccountRegistry'.hex()),
        #'TokenRegistry': '0x{:0<64s}'.format(b'TokenRegistry'.hex()),
        #'Faucet': '0x{:0<64s}'.format(b'TokenRegistry'.hex()),
        }

zero_address = '0x0000000000000000000000000000000000000000'


class CICRegistry:

    finalized = False
    contract = None
    chain_spec = None
    bancor_chain_spec = None
    accounts_chain_spec = None
    datadir = datadir
    w3 = None
    __chains_registry = {}

    __abi = None
    __bytecode = None
    
#    @staticmethod
#    def clear():
#        CICRegistry.finalized = False
#        CICRegistry.contract = None
#        CICRegistry.chain_spec = None
#        CICRegistry.bancor_chain_spec = None
#        CICRegistry.accounts_chain_spec = None
#
#
    @staticmethod
    def add_chain_registry(chain_registry):
        chain_str = chain_registry.chain()
        if CICRegistry.__chains_registry.get(chain_str) != None:
            logg.debug('skip add chain registry {}'.format(chain_str))
            raise ChainExistsError(chain_str)
        CICRegistry.__chains_registry[chain_str] = chain_registry


    @staticmethod
    def get_contract(chain_spec, identifier):
        chain_str = str(chain_spec)
        contract = CICRegistry.__chains_registry[chain_str].contracts.get(identifier)
        if contract == None:
            logg.info('contract {} not found in cache, checking on-chain {}'.format(identifier, chain_str))
            k = from_text(identifier)
            #contract_address = CICRegistry.contract.functions.addressOf(registry_identifiers[identifier]).call()
            contract_address = CICRegistry.contract.functions.addressOf(k).call()
            if contract_address == zero_address:
                raise ValueError('unknown contract identifier: {}'.format(identifier))
            abi = CICRegistry.abi(identifier)
            c = CICRegistry.w3.eth.contract(abi=abi, address=contract_address)
            CICRegistry.add_contract(chain_spec, c, identifier)
            contract = CICRegistry.get_contract(chain_spec, identifier)

        return contract


    @staticmethod
    def get_token(chain_spec, symbol):
        chain_str = str(chain_spec)
        token = None
        try:
            token = CICRegistry.__chains_registry[chain_str].get_token_by_symbol(symbol)
        except KeyError:
            c = CICRegistry.get_contract(chain_spec, 'TokenRegistry')
            fn = c.function('tokenSymbolIndex')
            token_address = fn(symbol).call()
            ct = w3.eth.contract(abi=CICRegistry.abi('ERC20'), address=token_address)
            token = CICRegistry.add_token(ct)
            #token = CICRegistry.__chains_registry[chain_str].get_token_by_symbol(symbol)

        return token



    @staticmethod
    def get_address(chain_spec, address):
        chain_str = str(chain_spec)
        contract = CICRegistry.__chains_registry[chain_str].addresses.get(address)
        if contract == None:
            raise UnknownContractError('{}:{}'.format(str(chain_spec), address))
        return contract
    

    @staticmethod
    def get_chain_registry(chain_spec):
        chain_str = str(chain_spec)
        return CICRegistry.__chains_registry[chain_str]


    @staticmethod
    def validate():
        c = CICRegistry.contract

        a = c.functions.addressOf(registry_identifiers['CICRegistry']).call()
        if a == zero_address:
            raise AttributeError('Registry address not set')

        a = c.functions.addressOf(registry_identifiers['BancorRegistry']).call()
        if a == zero_address:
            raise AttributeError('Bancor registry address not set')

        a = c.functions.addressOf(registry_identifiers['AccountRegistry']).call()
        if a == zero_address:
            raise AttributeError('Account registry address not set')


    @staticmethod
    def finalize(w3, address):
        if CICRegistry.finalized:
            raise AlreadyInitializedError('Already finalized')

        CICRegistry.w3 = w3

        abi = CICRegistry.abi()
        CICRegistry.contract = w3.eth.contract(abi=abi, address=address)
 
        CICRegistry.validate()

        c = CICRegistry.contract

        h = c.functions.chainOf(registry_identifiers['CICRegistry']).call()
        x = to_text(h)
        cr = ChainRegistry(x)
        try:
            CICRegistry.add_chain_registry(cr)
        except ChainExistsError:
            pass
        CICRegistry.chain_spec = ChainSpec.from_chain_str(x)
        d = c.functions.configSumOf(h).call()
        logg.warning('CICRegistry/{} chainconfig not checked: {}'.format(x, d.hex()))

        h = c.functions.chainOf(registry_identifiers['BancorRegistry']).call()
        x = to_text(h)
        cr = ChainRegistry(x)
        try:
            CICRegistry.add_chain_registry(cr)
        except ChainExistsError:
            pass
        CICRegistry.bancor_chain_spec = ChainSpec.from_chain_str(x)
        d = c.functions.configSumOf(h).call()
        logg.warning('BancorRegistry/{} chainconfig not checked: {}'.format(x, d.hex()))
        abi = CICRegistry.abi('RegistryClient')
        address = c.functions.addressOf(registry_identifiers['BancorRegistry']).call()
        bancor_registry_contract = w3.eth.contract(abi=abi, address=address)
        CICRegistry.__chains_registry[x].add_contract(bancor_registry_contract, 'BancorRegistry')


        h = c.functions.chainOf(registry_identifiers['AccountRegistry']).call()
        x = to_text(h)
        cr = ChainRegistry(x)
        try:
            CICRegistry.add_chain_registry(cr)
        except ChainExistsError:
            pass
        CICRegistry.accounts_chain_spec = ChainSpec.from_chain_str(x)
        d = c.functions.configSumOf(h).call()
        logg.warning('AccountRegistry/{} chainconfig not checked: {}'.format(x, d.hex()))
        abi = CICRegistry.abi('AccountRegistry')
        address = c.functions.addressOf(registry_identifiers['AccountRegistry']).call()
        account_registry_contract = w3.eth.contract(abi=abi, address=address)
        CICRegistry.__chains_registry[x].add_contract(account_registry_contract, 'AccountRegistry')

        CICRegistry.finalized = True


    @staticmethod
    def add_contract(chain_spec, contract, identifier):
        chain_str = str(chain_spec)
        c = CICRegistry.__chains_registry[chain_str].add_contract(contract, identifier)
        logg.info('registry added contract {}:{} ({})'.format(str(chain_spec), identifier, c.address()))


    @staticmethod
    def add_token(chain_spec, contract):
        chain_str = str(chain_spec)
        t = CICRegistry.__chains_registry[chain_str].add_token(contract)
        logg.info('registry added contract {}:{} ({})'.format(str(chain_spec), t.symbol(), t.address()))


    @staticmethod
    def add_address(chain_spec, address):
        chain_str = str(chain_spec)
        CICRegistry.__accounts_registry.add(address)


    @staticmethod
    def set_data_dir(d):
        CICRegistry.datadir = d
        logg.info('registry set data dir {}'.format(d))


    @staticmethod
    def abi(contract_identifier='CICRegistry'):
        if contract_identifier == 'CICRegistry':
            if CICRegistry.__abi == None:
                f = open(os.path.join(datadir, 'Registry.json'), 'r')
                CICRegistry.__abi = json.load(f)
                f.close()
            return CICRegistry.__abi
        f = open(os.path.join(CICRegistry.datadir, '{}.json'.format(contract_identifier)), 'r')
        abi = json.load(f)
        f.close()
        return abi


    @staticmethod
    def bytecode():
        if CICRegistry.__bytecode == None:
            f = open(os.path.join(datadir, 'Registry.bin'))
            CICRegistry.__bytecode = f.read()
            f.close()
        return CICRegistry.__bytecode


def to_text(b):
        b = b[:b.find(0)]
        return web3.Web3.toText(b)


def from_text(txt):
    return '0x{:0<64s}'.format(txt.encode('utf-8').hex())
