"""Set a new entry on CIC-registry

.. moduleauthor:: Louis Holbrook <dev@holbrook.no>
.. pgp:: 0826EDA1702D1E87C6E2875121D2E7BB88C2A746 

"""

# SPDX-License-Identifier: GPL-3.0-or-later

# standard imports
import os
import json
import argparse
import logging

# third-party imports
import web3

# local imports
from cic_registry import registry_identifiers
from cic_registry import from_text

logging.basicConfig(level=logging.WARNING)
logg = logging.getLogger()

logging.getLogger('web3').setLevel(logging.WARNING)
logging.getLogger('urllib3').setLevel(logging.WARNING)

script_dir = os.path.dirname(__file__)
data_dir = os.path.join(script_dir, '..', 'data')

argparser = argparse.ArgumentParser()
argparser.add_argument('-p', '--provider', dest='p', default='http://localhost:8545', type=str, help='Web3 provider url (http only)')
argparser.add_argument('-r', required=True, type=str, help='cic registry address')
argparser.add_argument('-s', required=True, type=str, help='chain spec')
argparser.add_argument('-d', default='0x00000000000000000000000000000000000000000000000000000000', type=str, help='chain config sha256 digest')
argparser.add_argument('-i', required=True, type=str, help='contract identifier key')
argparser.add_argument('-v', action='store_true', help='Be verbose')
argparser.add_argument('--abi-dir', dest='abi_dir', type=str, default=data_dir, help='Directory containing cic contract bytecode and abi (Default {})'.format(data_dir))
argparser.add_argument('address', type=str, help='contract address')
args = argparser.parse_args()


if args.v:
    logg.setLevel(logging.DEBUG)


def main():
    w3 = web3.Web3(web3.Web3.HTTPProvider(args.p))

    f = open(os.path.join(args.abi_dir, 'Registry.json'), 'r')
    abi = json.load(f)
    f.close()

    w3.eth.defaultAccount = w3.eth.accounts[0]
    c = w3.eth.contract(abi=abi, address=args.r)
    s_hex = args.s.encode('utf-8').hex()
    s_hex_pad = '0x{:0<64s}'.format(s_hex)

    #tx_transactable = c.functions.set(registry_identifiers[args.i], args.address, s_hex_pad, args.d)
    tx_transactable = c.functions.set(from_text(args.i), args.address, s_hex_pad, args.d)
    logg.debug('tx {}'.format(tx_transactable))
    tx = tx_transactable.transact()

    print(tx.hex())


if __name__ == '__main__':
    main()
