# standard imports
import os
import json

# local imports
from .base import (
        Data,
        data_dir,
        )


class Token(Data):
    """Encapsulates the token data used by the extension to deploy and/or register token and token related applications on chain.

    Token details (name, symbol etc) will be used to initialize the token settings when start is called. If load is called instead, any token detail parameters passed to the constructor will be overwritten by data stored in the settings.

    :param path: Settings directory path
    :type path: str
    :param name: Token name
    :type name: str
    :param symbol: Token symbol
    :type symbol: str
    :param precision: Token value precision (number of decimals)
    :type precision: int
    :param supply: Token supply (in smallest precision units)
    :type supply: int
    :param code: Bytecode for token chain application
    :type code: str (hex)
    """
    def __init__(self, path='.', name=None, symbol=None, precision=1, supply=0, code=None):
        super(Token, self).__init__()
        self.name = name
        self.symbol = symbol
        self.supply = supply
        self.precision = precision
        self.code = code
        self.extra_args = None
        self.path = path
        self.token_path = os.path.join(self.path, 'token.json')


    def load(self):
        """Load token data from settings.
        """
        super(Token, self).load()

        f = open(self.token_path, 'r')
        o = json.load(f)
        f.close()

        self.name = o['name']
        self.symbol = o['symbol']
        self.precision = o['precision']
        self.code = o['code']
        self.supply = o['supply']
        self.extra_args = o['extra']

        self.inited = True


    def start(self):
        """Initialize token settings from arguments passed to the constructor and/or template.
        """
        super(Token, self).load()

        token_template_file_path = os.path.join(data_dir, 'token_template_v{}.json'.format(self.version()))
        
        f = open(token_template_file_path)
        o = json.load(f)
        f.close() 

        o['name'] = self.name
        o['symbol'] = self.symbol
        o['precision'] = self.precision
        o['code'] = self.code
        o['supply'] = self.supply

        f = open(self.token_path, 'w')
        json.dump(o, f, sort_keys=True, indent="\t")
        f.close()


    def __str__(self):
        s = """name = {}
symbol = {}
precision = {}
""".format(self.name, self.symbol, self.precision)
        return s
