# standard imports
import urllib.request
import urllib.parse
import logging
import json

# external imports
from cic_types.models.tx import (
        TokenTx,
        TokenTxType,
        )

# local imports
from clicada.encode import tx_normalize
from clicada.error import ExpiredRecordError

logg = logging.getLogger(__name__)


class ResolvedTokenTx(TokenTx):

    def __init__(self):
        super(ResolvedTokenTx, self).__init__()
        self.source_token_name = None
        self.destination_token_name = None
        self.source_token_decimals = None
        self.destination_token_decimals = None
        self.symmetric = True
        self.sender_entity = None
        self.recipient_entity = None


    def resolve_tokens(self, token_store, show_decimals=False, update=False):
        (token_symbol, token_decimals) = token_store.by_address(self.source_token)
        self.source_token_decimals = token_decimals
        self.source_token_label = token_symbol
        token_value = self.to_value / (10 ** token_decimals)
        show_token_decimals = token_decimals
        if not show_decimals:
            show_token_decimals = 0
        fmt = '{:.' + str(show_token_decimals) + 'f}'
        self.from_value_label = fmt.format(token_value)

        if self.destination_token != self.source_token:
            self.symmetric = False
            (token_symbol, token_decimals) = token_store.by_address(self.destination_token)
            show_token_decimals = token_decimals

        self.destination_token_label = token_symbol
        self.destination_token_decimals = token_decimals
        token_value = self.to_value / (10 ** token_decimals)
        if not show_decimals:
            show_token_decimals = 0
        fmt = '{:.' + str(show_token_decimals) + 'f}'

        self.to_value_label = fmt.format(token_value)


    def resolve_stored_entity(self, user_store, address, update=False):
        if update:
            return None
        address = tx_normalize.wallet_address(address)
        try:
            v = user_store.get(address)
            v = json.loads(v)
            return user_store.metadata_to_person(v)
        except FileNotFoundError:
            return None
        except ExpiredRecordError:
            return None


    def resolve_sender_entity(self, user_store, update=False):
        v = self.resolve_stored_entity(user_store, self.sender, update=update)
        if v != None:
            return v
        if self.tx_type == TokenTxType.faucet_giveto.value:
            return 'FAUCET'
        r = user_store.by_address(self.sender)
        return str(r)


    def resolve_recipient_entity(self, user_store, update=False):
        v = self.resolve_stored_entity(user_store, self.recipient, update=update)
        if v != None:
            return v
        r = user_store.by_address(self.recipient, update=update)
        return str(r)


    def resolve_entities(self, user_store, update=False):
        self.sender_label = self.resolve_sender_entity(user_store, update=update)
        self.recipient_label = self.resolve_recipient_entity(user_store, update=update)


    def resolve(self, token_store, user_store, show_decimals=False, update=False):
        self.resolve_tokens(token_store, show_decimals, update=update)
        self.resolve_entities(user_store, update=update)


    def __str__(self):
        if self.symmetric:
            return '{}\t{} => {}\t{} {}'.format(
                    self.date_block_label,
                    self.sender_label,
                    self.recipient_label,
                    self.destination_token_label,
                    self.to_value_label,
            )


class TxGetter:

    def __init__(self, cache_url, limit=0):
        self.cache_url = cache_url
        self.limit = limit

        if limit == 0:
            url = urllib.parse.urljoin(self.cache_url, 'defaultlimit')
            r = urllib.request.urlopen(url)
            self.limit = json.load(r)
            logg.debug('set limit {} from {}'.format(self.limit, self.cache_url))


    def get(self, address, offset=0, limit=0):
        url = urllib.parse.urljoin(self.cache_url, '/txa/user/{}/{}/{}'.format(address, limit, offset))
        r = urllib.request.urlopen(url)
        return json.load(r)
