# standard imports
import threading
import logging

# external imports
from hexathon import strip_0x

# local imports
from clicada.tx import ResolvedTokenTx
from clicada.token import token_balance
from clicada.error import MetadataNotFoundError

logg = logging.getLogger(__name__)


class MetadataResolverWorker(threading.Thread):
    
    def __init__(self, wallet_address, ctrl, user_address_store):
        self.user_address_store = user_address_store
        self.wallet_address = wallet_address
        self.ctrl = ctrl
        super(MetadataResolverWorker, self).__init__()
        

    def run(self):
        self.ctrl.notify('resolve metadata for {}'.format(self.wallet_address))
        try:
            self.user_address_store.by_address(self.wallet_address)
        except MetadataNotFoundError:
            logg.info('failed metadata lookup for {}'.format(self.wallet_address))


class TxResolverWorker(threading.Thread):
    def __init__(self, tx_hash, tx_src, ctrl, tx_store, token_store, user_address_store, token_queue, tx_queue, show_decimals=True, update=None):
        self.tx_hash = tx_hash
        self.tx_src = tx_src
        self.ctrl = ctrl
        self.token_store = token_store
        self.user_address_store = user_address_store
        self.show_decimals = show_decimals
        self.update = update
        self.token_queue = token_queue
        self.tx_store = tx_store
        self.tx_queue = tx_queue
        super(TxResolverWorker, self).__init__()


    def run(self):
        self.ctrl.notify('resolve details for tx {}'.format(self.tx_hash))
        tx = ResolvedTokenTx.from_dict(self.tx_src)
        tx.resolve(self.token_store, self.user_address_store, show_decimals=self.show_decimals, update=self.update, lookup=False)
        self.tx_store.put(self.tx_hash, str(self.tx_src), overwrite=self.ctrl.get('_FORCE_ALL'))
        self.tx_queue.put(tx)


class TokenResolverWorker(threading.Thread):

    def __init__(self, wallet_address, ctrl, token_store, in_queue, out_queue):
        super(TokenResolverWorker, self).__init__()
        self.ctrl = ctrl
        self.token_store = token_store
        self.in_queue = in_queue
        self.out_queue = out_queue
        self.seen_tokens = {}
        self.wallet_address = strip_0x(wallet_address)


    def run(self):
        while True:
            token_address = self.in_queue.get()
            if token_address == None:
                logg.debug('token resolver end')
                self.out_queue.put_nowait(None)
                return
            token_address = strip_0x(token_address)
            if self.seen_tokens.get(token_address) != None:
                continue
            logg.debug('resolve token {}'.format(token_address))
            self.ctrl.notify('resolve token {}'.format(token_address))
            (token_symbol, token_decimals) = self.token_store.by_address(token_address)
            self.seen_tokens[token_address] = token_address
            self.ctrl.notify('get token balance for {} => {}'.format(token_symbol, self.seen_tokens[token_address]))
            balance = token_balance(self.ctrl.chain(), self.ctrl.conn(), self.seen_tokens[token_address], self.wallet_address)
            fmt =  '{:.' + str(token_decimals) + 'f}'
            decimal_balance = fmt.format(balance / (10 ** token_decimals))
            logg.debug('token balance for {} ({}) is {}'.format(token_symbol, token_address, decimal_balance))
            self.out_queue.put((token_symbol, decimal_balance,))
