# standard imports
import hashlib
import logging
import json

# external imports
from Crypto.Cipher import AES
from Crypto.Util import Counter
import sha3

logging.basicConfig(level=logging.DEBUG)
logg = logging.getLogger()

algo_keywords = [
    'aes-128-ctr',
        ]
hash_keywords = [
    'scrypt'
        ]


class Hashes:

    @staticmethod
    def from_scrypt(kdfparams, passphrase=''):

        dklen = int(kdfparams['dklen']) 
        n = int(kdfparams['n']) 
        p = int(kdfparams['p']) 
        r = int(kdfparams['r']) 
        salt = bytes.fromhex(kdfparams['salt'])

        return hashlib.scrypt(passphrase.encode('utf-8'), salt=salt,n=n, p=p, r=r, maxmem=1024*1024*1024, dklen=dklen)


class Ciphers:

    aes_128_block_size = 1 << 7

    def decrypt_aes_128_ctr(ciphertext, decryption_key, iv):
        ctr = Counter.new(Ciphers.aes_128_block_size, initial_value=iv) #
        cipher = AES.new(decryption_key[:16], AES.MODE_CTR, counter=ctr)
        plaintext = cipher.decrypt(ciphertext)
        return plaintext


def from_dict(o, passphrase=''):

    cipher = o['crypto']['cipher']
    if cipher not in algo_keywords:
        raise NotImplementedError('cipher "{}" not implemented'.format(cipher))

    kdf = o['crypto']['kdf']
    if kdf not in hash_keywords:
        raise NotImplementedError('kdf "{}" not implemented'.format(kdf))

    m = getattr(Hashes, 'from_{}'.format(kdf.replace('-', '_'))) 
    decryption_key = m(o['crypto']['kdfparams'], passphrase)

    mac = bytes.fromhex(o['crypto']['mac'])
    iv_bytes = bytes.fromhex(o['crypto']['cipherparams']['iv'])
    iv = int.from_bytes(iv_bytes, "big")
    ciphertext_bytes = bytes.fromhex(o['crypto']['ciphertext'])
    
    # check mac
    h = sha3.keccak_256()
    h.update(decryption_key[16:])
    h.update(ciphertext_bytes)
    z = h.digest()
    assert z == mac

    m = getattr(Ciphers, 'decrypt_{}'.format(cipher.replace('-', '_')))
    pk = m(ciphertext_bytes, decryption_key, iv)
    return pk


def from_file(filepath, passphrase=''):

    f = open(filepath, 'r')
    o = json.load(f)
    f.close()

    return from_dict(o, passphrase)
