# standard imports
import os
import logging
import time
from urllib import request
import hashlib

# third-party imports
import yaml
import eth_keys

# local imports
from .base import Retriever
from .error import SessionError
from .error import ChallengeError
from .error import TokenExpiredError
from .session import Session
from .challenge import AuthChallenge

logg = logging.getLogger()


def signature_patch_v(signature_bytes):
    """Python package eth-keys returns recovery byte 00-01 instead of the eth standard 27-28. This patches any incoming 'correct' signature to the local 'incorrect' one.

    :params signature_bytes: Raw signature
    :type signature_bytes: bytes
    :return: Patched signature
    :rtype: Bytes
    """
    m = signature_bytes[64] % 27
    patched_signature_bytes = signature_bytes[:64] + bytes([m])
    return patched_signature_bytes

def source_hash(ip, nonce):
    """Generate a unique ip/nonce key for auth challenge index

    :param ip: Textual ip address
    :type ip: str
    :param nonce: Arbitrary byte string (typically challenge value)
    :type nonce: bytes
    :return: Digest to use as key
    :rtype: bytes
    """
    h = hashlib.sha256()
    h.update(ip.encode('utf-8'))
    h.update(nonce)
    k = h.digest()
    return k


class SimpleRetriever(Retriever):
    """A single-url retriever for the ACL list.

    Will attempt to retrieve an ACL list from a file matching the 0x-prefixed hex string at the base_url given at construction time.

    :param base_url: Base HTTP url for ACL document retrieval.
    :type base_url: str, url
    :param decrypter: Callback function to process decrypted and/or signed data. 
    :type decrypter: function, must accept binary data as single argument.
    """
    def __init__(self, config, decrypter):
        self.base_url = config.get('ECUTH_BASE_URL')
        self.name = config.get('EIP712_NAME')
        self.version = config.get('EIP712_VERSION')
        self.chain_id = int(config.get('ECUTH_CHAIN_ID'))
        self.decrypter = decrypter
        self.session = {}
        self.auth = {}
        self.session_reverse = {}
        self.challenge_filter = []
        self.challenge_filter_index = {}
        logg.info('SimpleRetriever initialized with base_url {}'.format(self.base_url))


    def add_challenge_filter(self, filter_method, name=None):
        if not callable(filter_method):
            raise ValueError('filter must be callable')
        if name == None:
            name = filter_method.__name__
        if self.challenge_filter_index.get(name) != None:
            idx = self.challenge_filter_index[name]
            logg.info('resetting challenge filter {} on index {}'.format(name, idx))
            self.challenge_filter[idx][1] = filter_method
        else:
            self.challenge_filter_index[name] = len(self.challenge_filter)
            self.challenge_filter.append((name, filter_method,))


    def _fetch(self, address):
        """Uses instance member base_url to retrieve ACL document with http.
        
        :param address: Ethereum address of user
        :type address: str, 0x-hex
        :raises urllib.error.URLError:
        :raises http.client.RemoteDisconnected:
        :return: Retrieved data
        :rtype: bytes
        """
        url = os.path.join(self.base_url, address)
        req = request.Request(url)
        req.add_header('Accept', 'application/octet-stream')
        response = request.urlopen(req)
        return response.read()


    def clear(self, address):
        """Remove all session data.

        :param address: Ethereum address of user
        :type address: str, 0x-hex
        """
        if self.auth.get(address) != None:
            del self.auth[address]
        if self.session.get(address) != None:
            del self.session[address]


    # TODO: use pubkey instead of address, and use ec library directly
    def challenge(self, ip):
        """Generate a new challenge string for user address. This must be signed by the corresponding private key of the address.

        :param address: Ethereum address of user
        :type address: str, 0x-hex
        :return: Challenge token
        """
        c = AuthChallenge(ip, self.challenge_filter)
        (nonce, expire) = c.request()
        k = source_hash(ip, nonce)
        self.auth[k] = c
        logg.info('generated new challenge {} expires {}'.format(nonce.hex(), expire))
        return (nonce, expire,)


    def _address_from_challenge_response(self, challenge_key, challenge_original, signature):
        """Verifies and recovers public key and address for the signature and challenge. 

        :param challenge: Challenge signed by user
        :type challenge: bytes
        :param signature: Challenge signature 
        :type signature: bytes
        :raises ecuth.error.ChallengeError: Challenge not found for address, or signature could not be verified.
        :return: Recovered address
        :rtype: bytes
        """
        #logg.debug('signature {} challenge {}'.format(signature.hex(), challenge_original.hex()))
        challenge_instance = self.auth[challenge_key]
        challenge_compare = challenge_instance.apply_filters(challenge_original)
        challenge_correct = challenge_instance.challenge

        # Make sure signature has v byte understood by eth-keys package
        signature_patched_bytes = signature_patch_v(signature)
        signature_patched = eth_keys.datatypes.Signature(signature_bytes=signature_patched_bytes)

        pubkey = signature_patched.recover_public_key_from_msg(challenge_compare)
        address = pubkey.to_checksum_address()
        if challenge_correct == None:
            raise ChallengeError('no challenge exists for address {}'.format(address))
        if challenge_correct != challenge_original:
            raise ChallengeError('challenge mismatch for address {}'.format(address))
        return address


    def load(self, ip, challenge, signature):
        """Retrieves ACL and initializes session on a successful validation of challenge signature.

        :param challenge: Challenge signed by user
        :type challenge: bytes
        :param signature: Challenge signature 
        :type signature: bytes
        :raises ecuth.error.ChallengeError: Challenge not found for address, or signature could not be verified.
        :raises urllib.error.URLError: Invalid fetch URL 
        :raises http.client.RemoteDisconnected: HTTP Connection exception
        :return: Refresh token and authentication token, respectively
        :rtype: tuple of bytestrings
        """
        # retrieve the challenge value
        challenge_key = source_hash(ip, challenge)
        challenge_object = self.auth[challenge_key]

        address = self._address_from_challenge_response(challenge_key, challenge, signature)
        logg.debug('eip712 challenge: {} address {}'.format(challenge.hex(), address))

        # at this point successful authentication, challenge can be removed.
        self.auth[challenge_key].clear()

        data_encrypted = self._fetch(address)
        data_plaintext = self.decrypter(data_encrypted)
        y = yaml.load(data_plaintext, Loader=yaml.FullLoader)
       
        session = Session(address, y['level'], y['items'])
        self.session[address] = session
        self.renew(address, session.refresh)

        logg.debug('added session {}'.format(session))

        return (
                self.session[address].refresh,
                self.session[address].auth,
                self.session[address].auth_expire,
        )

    
    def get(self, address, item):
        """Retrieves the access value for a specific ACL item.

        The access values are:

        - 0x04 read
        - 0x02 write

        :param address: Ethereum address of user
        :type address: str, 0x-hex
        :param item: ACL entry to retrieve
        :type item: str
        :raises ecuth.error.SessionError: Session does not exist
        :raises ecuth.error.TokenExpiredError: Auth token expired (must refresh)
        :raises ValueError: ACL entry does not exist
        :return: Access value
        :rtype: int
        """
        session = self.session.get(address)
        if session == None:
            raise SessionError('no session for {}'.format(address))
        if not self.session[address].valid():
            raise TokenExpiredError(address) 
        axx = session.items[item]
        return axx


    def renew(self, address, refresh_token):
        """Renews an expired auth token.

        :param address: Ethereum address of user
        :type address: str, 0x-hex
        :raises ecuth.error.SessionExpiredError: Refresh token expired (must restart challenge)
        :return: New auth token
        :rtype: bytes
        """
        old_token = self.session[address].auth
        new_token = self.session[address].renew(refresh_token)
        self.session_reverse[new_token] = address
        if old_token != None:
            del self.session_reverse[old_token]
        return new_token


    def read(self, address, item):
        """Check whether the ACL item defines read access.

        :param address: Ethereum address of user
        :type address: str, 0x-hex
        :param item: ACL entry to retrieve
        :type item: str
        :raises ValueError: ACL entry does not exist
        :return: Result
        :rtype: boolean
        """
        return self.get(address, item) & 4 > 0


    def write(self, address, item):
        """Check whether the ACL item defines write access.

        :param address: Ethereum address of user
        :type address: str, 0x-hex
        :param item: ACL entry to retrieve
        :type item: str
        :raises ValueError: ACL entry does not exist
        :return: Result
        :rtype: boolean
        """
        return self.get(address, item) & 2 > 0



    def check(self, token):
        """Check whether given auth token is still valid.
        
        :raises ValueError: Session does not exist
        """
        address = self.session_reverse[token]
        current_token = self.session[address].auth
        if token != current_token:
            raise SessionError('invalid token {} for address {}'.format(token, address))
