# standard imports
import uuid
import logging
import time
import eth_keys
import hashlib

logg = logging.getLogger()

DEFAULT_CHALLENGE_EXPIRE = 10


def source_hash(ip, data):
    """Creates a unique index value for a combination of (client) ip and challenge data.

    :param ip: IP address in text format, translated to bytes
    :type ip: bytes
    :param data: The data represented by the index
    :type data: bytes
    :return: Preimage of ip and data
    :rtype: bytes
    """
    h = hashlib.sha256()
    h.update(ip)
    h.update(data)
    k = h.digest()
    return k


class AuthChallenge:
    """Minimal convenience object representing an authentication challenge.

    :param ip: IP address of client
    :type p: str
    :param filters: List of filter methods to be executed on a challenge when apply_filters is called.
    :type filters: function, taking a single byte-string argument, returning a byte-string.
    """
    def __init__(self, ip, filters):
        self.ip = ip
        self.challenge = None
        self.challenge_expire = 0
        self.filter = filters


    def request(self):
        """Creates a new challenge.

        :return: Challenge value
        :rtype: bytes
        """
        uu = uuid.uuid4()
        self.challenge = uu.bytes
        self.challenge_expire = time.time() + DEFAULT_CHALLENGE_EXPIRE
        return (self.challenge, self.challenge_expire,)


    # TODO: the filters should be applied on the stored challenge. If a stateless transformation is needed, this method should be made static instead.
    def apply_filters(self, s):
        """Executes the challenge transformation filters on the given challenge.

        This does not modify the original challenge already stored in the instance.
        
        :param s: Challenge to transform.
        :type s: bytes
        :return: Fitered challenge value
        :rtype: bytes
        """
        for f in self.filter:
            logg.debug('applying filter {}'.format(f[0]))
            s = f[1](s)
        return s


    def clear(self):
        """Clears existing challenge data.
        """
        self.challenge = None
        self.challenge_expire = 0
