# standard imports
import os
import logging
import time
import hashlib

# third-party imports
import eth_keys

# local imports
from .digest import DigestRetriever
from .error import ChallengeError
from .session import Session
from .challenge import AuthChallenge

logg = logging.getLogger(__name__)


def signature_patch_v(signature_bytes):
    """Python package eth-keys returns recovery byte 00-01 instead of the eth standard 27-28. This patches any incoming 'correct' signature to the local 'incorrect' one.

    :params signature_bytes: Raw signature
    :type signature_bytes: bytes
    :return: Patched signature
    :rtype: Bytes
    """
    m = signature_bytes[64] % 27
    patched_signature_bytes = signature_bytes[:64] + bytes([m])
    return patched_signature_bytes


def source_hash(ip, nonce):
    """Generate a unique ip/nonce key for auth challenge index

    :param ip: Textual ip address
    :type ip: str
    :param nonce: Arbitrary byte string (typically challenge value)
    :type nonce: bytes
    :return: Digest to use as key
    :rtype: bytes
    """
    h = hashlib.sha256()
    h.update(ip.encode('utf-8'))
    h.update(nonce)
    k = h.digest()
    return k


class EthereumRetriever(DigestRetriever):
    """A single-url retriever for the ACL list.

    Will attempt to retrieve an ACL list from a file matching the 0x-prefixed hex string at the base_url given at construction time.

    :param fetcher: Callback function to retrieve and decrypt acl data
    :type fetcher: function, must accept binary identity identifier as single argument.
    """
    def __init__(self, fetcher):
        super(EthereumRetriever, self).__init__(fetcher)
        self.auth = {}
        self.challenge_filter = []
        self.challenge_filter_index = {}


    def add_challenge_filter(self, filter_method, name=None):
        if not callable(filter_method):
            raise ValueError('filter must be callable')
        if name == None:
            name = filter_method.__name__
        if self.challenge_filter_index.get(name) != None:
            idx = self.challenge_filter_index[name]
            logg.info('resetting challenge filter {} on index {}'.format(name, idx))
            self.challenge_filter[idx][1] = filter_method
        else:
            self.challenge_filter_index[name] = len(self.challenge_filter)
            self.challenge_filter.append((name, filter_method,))


    def clear(self, address):
        """Remove all session data.

        :param address: Ethereum address of user
        :type address: str, 0x-hex
        """
        if self.auth.get(address) != None:
            del self.auth[address]
        super(SimpleRetriever, self).clear(address)


    # TODO: use pubkey instead of address, and use ec library directly
    def challenge(self, ip):
        """Generate a new challenge string for user address. This must be signed by the corresponding private key of the address.

        :param address: Ethereum address of user
        :type address: str, 0x-hex
        :return: Challenge token
        """
        c = AuthChallenge(ip, self.challenge_filter)
        (nonce, expire) = c.request()
        k = source_hash(ip, nonce)
        self.auth[k] = c
        logg.info('generated new challenge {} expires {}'.format(nonce.hex(), expire))
        return (nonce, expire,)


    def _address_from_challenge_response(self, challenge_key, challenge_original, signature):
        """Verifies and recovers public key and address for the signature and challenge. 

        :param challenge: Challenge signed by user
        :type challenge: bytes
        :param signature: Challenge signature 
        :type signature: bytes
        :raises ecuth.error.ChallengeError: Challenge not found for address, or signature could not be verified.
        :return: Recovered address
        :rtype: bytes
        """
        challenge_instance = self.auth[challenge_key]
        challenge_compare = challenge_instance.apply_filters(challenge_original)
        challenge_correct = challenge_instance.challenge

        # Make sure signature has v byte understood by eth-keys package
        signature_patched_bytes = signature_patch_v(signature)
        signature_patched = eth_keys.datatypes.Signature(signature_bytes=signature_patched_bytes)

        pubkey = signature_patched.recover_public_key_from_msg(challenge_compare)
        address = pubkey.to_checksum_address()
        logg.debug('address >>>>>>> {}'.format(address))
        if challenge_correct == None:
            raise ChallengeError('no challenge exists for address {}'.format(address))
        if challenge_correct != challenge_original:
            raise ChallengeError('challenge mismatch for address {}'.format(address))
        address_bytes = bytes.fromhex(address[2:])
        return address_bytes


    def response(self, ip, challenge, signature):
        """Validation of challenge signature.

        :param challenge: Challenge signed by user
        :type challenge: bytes
        :param signature: Challenge signature 
        :type signature: bytes
        :raises ecuth.error.ChallengeError: Challenge not found for address, or signature could not be verified.
        :raises urllib.error.URLError: Invalid fetch URL 
        :raises http.client.RemoteDisconnected: HTTP Connection exception
        :return: Refresh token and authentication token, respectively
        :rtype: tuple of bytestrings
        """
        # retrieve the challenge value
        challenge_key = source_hash(ip, challenge)
        challenge_object = self.auth[challenge_key]

        address = self._address_from_challenge_response(challenge_key, challenge, signature)
        logg.debug('eip712 challenge: {} address {}'.format(challenge.hex(), address.hex()))

        # at this point successful authentication, challenge can be removed.
        self.auth[challenge_key].clear()
        return address
