# standard imports
import logging

# third-party imports
import yaml
from .base import Retriever
from ecuth.session import Session
from ecuth.error import SessionError, TokenExpiredError

logg = logging.getLogger(__name__)


def noop_decrypter(s):
    logg.warning('using generic noop decrypter, that is probably not a good idea')
    return s


class DigestRetriever(Retriever):
    def __init__(self, fetcher):
        self.fetcher = fetcher
        self.session = {}
        self.session_reverse = {}


    def clear(self, address):
        if self.session.get(address) != None:
            del self.session[address]


    def load(self, identity):
        """Retrieves ACL and initializes session 
        """
        data = self.fetcher(identity)
        y = yaml.load(data, Loader=yaml.FullLoader)
       
        session = Session(identity, y['level'], y['items'])
        self.session[identity] = session
        self.renew(identity, session.refresh)

        logg.debug('added session {}'.format(session))

        return (
                self.session[identity].refresh,
                self.session[identity].auth,
                self.session[identity].auth_expire,
        )


    def get(self, address, item):
        """Retrieves the access value for a specific ACL item.

        The access values are:

        - 0x04 read
        - 0x02 write

        :param address: Ethereum address of user
        :type address: str, 0x-hex
        :param item: ACL entry to retrieve
        :type item: str
        :raises ecuth.error.SessionError: Session does not exist
        :raises ecuth.error.TokenExpiredError: Auth token expired (must refresh)
        :raises ValueError: ACL entry does not exist
        :return: Access value
        :rtype: int
        """
        session = self.session.get(address)
        if session == None:
            raise SessionError('no session for {}'.format(address))
        if not self.session[address].valid():
            raise TokenExpiredError(address) 
        axx = session.items[item]
        return axx


    def renew(self, address, refresh_token):
        """Renews an expired auth token.

        :param address: Ethereum address of user
        :type address: str, 0x-hex
        :raises ecuth.error.SessionExpiredError: Refresh token expired (must restart challenge)
        :return: New auth token
        :rtype: bytes
        """
        old_token = self.session[address].auth
        new_token = self.session[address].renew(refresh_token)
        self.session_reverse[new_token] = address
        if old_token != None:
            del self.session_reverse[old_token]
        return new_token


    def read(self, address, item):
        """Check whether the ACL item defines read access.

        :param address: Ethereum address of user
        :type address: str, 0x-hex
        :param item: ACL entry to retrieve
        :type item: str
        :raises ValueError: ACL entry does not exist
        :return: Result
        :rtype: boolean
        """
        return self.get(address, item) & 4 > 0


    def write(self, address, item):
        """Check whether the ACL item defines write access.

        :param address: Ethereum address of user
        :type address: str, 0x-hex
        :param item: ACL entry to retrieve
        :type item: str
        :raises ValueError: ACL entry does not exist
        :return: Result
        :rtype: boolean
        """
        return self.get(address, item) & 2 > 0



    def check(self, token):
        """Check whether given auth token is still valid.
        
        :raises ValueError: Session does not exist
        """
        address = self.session_reverse[token]
        current_token = self.session[address].auth
        if token != current_token:
            raise SessionError('invalid token {} for address {}'.format(token, address))
