# standard imports
import time
import uuid

# local imports
from .error import SessionExpiredError
from .error import SessionError

DEFAULT_AUTH_EXPIRE = 60
DEFAULT_REFRESH_EXPIRE = 60 * 60 * 24 * 30


class Session:
    """State of authentication session and ACL data.

    :param address: Ethereum address of user
    :type address: str, 0x-hex
    :param level: Authority level of user
    :type level: int
    :param items: ACL entries for user
    :type items: dict
    """
    def __init__(self, identity, level, items):
        self.identity = identity
        self.level = level
        self.items = items
        uu = uuid.uuid4()
        self.refresh = uu.bytes
        self.refresh_expire = time.time() + DEFAULT_REFRESH_EXPIRE
        self.auth = None
        self.auth_expire = 0.0


    def renew(self, refresh):
        """Renew auth token.

        :param refresh: Refresh token
        :type refresh: bytes
        :raises ecuth.error.SessionError: Refresh token unknown
        :raises ecuth.error.SessionExpiredError: Refresh token expired (must restart challenge)
        :return: New auth token
        :rtype: bytes
        """
        if self.refresh != refresh:
            raise SessionError('invalid refresh token for {}'.format(self.identity))
        if self.refresh_expire < time.time():
            raise SessionExpiredError(self.identity)
        uu = uuid.uuid4()
        self.auth = uu.bytes
        self.auth_expire = time.time() + DEFAULT_AUTH_EXPIRE
        return self.auth


    def valid(self):
        """Check if auth token is currently valid.

        :returns: Valid or not
        :rtype: boolean
        """
        return self.auth_expire > time.time()


    def __str__(self):
        f = """level: {}
identity: {}
token: {}
expires: {}
items:
"""
        d = f.format(
            self.level,
            self.identity.hex(),
            self.auth.hex(),
            self.auth_expire,
        )

        for k in self.items.keys():
            d += "\t{}: {}\n".format(k, self.items[k])
        
        return d
