# standard imports
import logging
import tempfile
import urllib.request

# third-party imports
import gnupg

# local imports
from ecuth.challenge import ChallengeRetriever
from ecuth.error import AlienMatterError

logg = logging.getLogger()


class PGPRetriever(ChallengeRetriever):


    def __init__(self, fetcher, parser, trusted_keys, gnupg_home=None):
        super(PGPRetriever, self).__init__(fetcher, parser, self._fingerprint_from_challenge_response)
        self.gpg = gnupg.GPG(gnupghome=gnupg_home)
        import_result = self.gpg.import_keys(trusted_keys)
        self.trusted = []
        self.auth_keys = []
        for r in import_result.results:
            if r['fingerprint'] == None:
                logg.debug('skipping invalid key')
                continue
            fingerprint = r['fingerprint']
            self.trusted.append(fingerprint)
            logg.info('added trusted acl pgp signer key {}'.format(fingerprint))
   

    def verify_import(self, export):
        r = self.gpg.verify(export)
        if r.status != 'signature valid':
            return False
        return r.fingerprint in self.trusted


    def import_keys(self, export):
        if not self.verify_import(export):
            raise AlienMatterError('pgp public key bundle')
        export_content = self.gpg.decrypt(export)
        r = self.gpg.import_keys(str(export_content))
        for k in r.results:
            if k['fingerprint'] == None:
                logg.debug('skipping invalid auth key')
                continue
            logg.info('imported auth pgp key {}'.format(k['fingerprint']))
            self.auth_keys.append(k['fingerprint'])


    def _fingerprint_from_challenge_response(self, challenge, signature):
        fn = tempfile.mkstemp()
        f = open(fn[1], 'wb')
        f.write(signature)
        f.close()

        v = None
        try: 
            v = self.gpg.verify_data(fn[1], challenge)
        except Exception as e:
            logg.error('error verifying {}'.format(e))
            return None

        if not v.valid:
            logg.error('signature not valid {}'.format(v.fingerprint))
            return None

        logg.debug('signature valid!')
        return bytes.fromhex(v.fingerprint)
