# standard imports
import logging
import base64
import hashlib
from urllib.parse import urlparse

logg = logging.getLogger()

signature_algorithm_ids = {
        'secp256k1': 42,
        }


def validate_origin(origin):
    parsed_origin = urlparse(origin)
    if parsed_origin.scheme not in ['http', 'https']:
        raise ValueError('invalid origin {}, not http(s)'.format(origin))
    if parsed_origin.port == None:
        raise ValueError('port missing from origin')
    return origin


class Hoba:

    def __init__(self, origin, realm, signature_algorithm='secp256k1'):
        try:
            self.alg = signature_algorithm_ids[signature_algorithm]
        except KeyError:
            sig_ids = signature_algorithm_ids.keys()
            if signature_algorithm not in sig_ids:
                raise NotImplementedError('allowed signature algorithms: {}'.format(sig_ids))
        self.origin = validate_origin(origin)
        self.realm = realm
        self.nonce = None
        self.kid = None
        self.challenge = None
        self.signature = None


    
    def parse(self, s):
        fields = s.split('.')
        self.nonce = base64.b64decode(fields[0])
        self.kid = base64.b64decode(fields[1])
        self.challenge = base64.b64decode(fields[2])
        self.signature = base64.b64decode(fields[3])
        logg.debug('parsed hoba nonce {}'.format(self.nonce.hex()))
        logg.debug('parsed hoba kid {}'.format(self.kid.hex()))
        logg.debug('parsed hoba challenge {}'.format(self.challenge.hex()))
        logg.debug('parsed hoba signature {}'.format(self.signature.hex()))


    def to_be_signed(self):
        nonce_bytes = base64.b64encode(self.nonce)
        nonce = nonce_bytes.decode('utf-8')
        kid_bytes = base64.b64encode(self.kid)
        kid = kid_bytes.decode('utf-8')
        challenge_bytes = base64.b64encode(self.challenge)
        challenge = challenge_bytes.decode('utf-8')

        s = ''
        for f in [nonce, str(self.alg), self.origin, self.realm, kid, challenge]:
            s += '{}:{}'.format(str(len(f)), f)

        return s
