# standard imports
import logging
import ctypes

# local imports
from . import (
    LIBRLP_RLP_MAX_LIST_DEPTH,
    librlp,
        )

logg = logging.getLogger().getChild(__name__)


class RLPEncoder:

    __nullptr = ctypes.POINTER(ctypes.c_void_p)()

    def __init__(self, buffer_size):

        class RLPEncoderBackend(ctypes.Structure):

            _fields_ = [
                    ('buf', ctypes.POINTER(ctypes.c_char * buffer_size)),
                    ('alloc', ctypes.c_char),
                    ('depth', ctypes.c_int),
                    ('size', ctypes.c_int),
                    ('state', ctypes.c_int),
                    ('list_ptr', ctypes.POINTER(ctypes.POINTER(ctypes.c_char)) * LIBRLP_RLP_MAX_LIST_DEPTH),
                    ('ptr', ctypes.POINTER(ctypes.c_char)),
                    ]
    
        self.buffer_size = buffer_size
        self.backend = RLPEncoderBackend()
        self.encoder = ctypes.pointer(self.backend)


    def __del__(self):
        logg.debug('free')
        librlp.rlp_free(self.encoder)


    def encode_item(self, v):
        if isinstance(v, list):
            librlp.rlp_descend(self.encoder)
            for e in v:
                self.encode_item(e)
            librlp.rlp_ascend(self.encoder)

        else:
            b = (ctypes.c_char * len(v))(*v)
            librlp.rlp_add(self.encoder, len(v), b)

        return self.backend.size


    def encode(self, v):
        librlp.rlp_init(self.encoder, self.buffer_size, self.__nullptr)
        r = self.encode_item(v)

        return bytes(self.backend.buf.contents[:r])
