# standard imports
import os
import logging
import tempfile

# external imports
import cic_base.log
import cic_base.argparse
import cic_base.config
import cic_base.signer
from chainlib.eth.connection import EthHTTPConnection
from chainlib.eth.nonce import (
        RPCNonceOracle,
        OverrideNonceOracle,
        )
from chainlib.eth.gas import (
        RPCGasOracle,
        OverrideGasOracle,
        )
from chainlib.chain import ChainSpec

# local imports
from sarafu_faucet.factory import MinterFaucet

script_dir = os.path.realpath(os.path.dirname(__file__))

logg = cic_base.log.create()

def add_token_args(argparser):
    argparser.add_argument('-w', action='store_true', help='Wait for the last transaction to be confirmed')
    argparser.add_argument('-ww', action='store_true', help='Wait for every transaction to be confirmed')
    argparser.add_argument('--gas-price', type=int, dest='gas_price', help='Override gas price')
    argparser.add_argument('-d', action='store_true', help='Dump RPC calls to terminal and do not send')
    argparser.add_argument('--nonce', type=int, help='Override transaction nonce')
    argparser.add_argument('--contract-address', '-a', dest='a', required=True, type=str, help='faucet address')
    argparser.add_argument('recipient', type=str, help='faucet gift recipient address')
    return argparser


argparser = cic_base.argparse.create(script_dir, include_args=['provider', 'keystore_file', 'chain_spec', 'config_dir'])
argparser = cic_base.argparse.add(argparser, add_token_args, 'token')
args = cic_base.argparse.parse(argparser, logg)

block_last = args.w
block_all = args.ww

config = cic_base.config.create(args.c, args, arg_overrides={
    'p': 'ETH_PROVIDER',
    'i': 'CIC_CHAIN_SPEC',
    }
    )
config.add(args.y, '_KEYSTORE_FILE', True)
config.add(args.a, '_CONTRACT_ADDRESS', True)
config.add(args.recipient, '_RECIPIENT', True)
config.add(args.gas_price, '_GAS_PRICE', True)
config.add(args.nonce, '_NONCE', True)
config.add(args.d, '_DUMMY', True)

cic_base.config.log(config)

def main():

    (signer_address, signer) = cic_base.signer.from_keystore(config.get('_KEYSTORE_FILE'))

    conn = EthHTTPConnection(config.get('ETH_PROVIDER'))
    nonce_oracle = None
    if args.nonce != None:
        nonce_oracle = OverrideNonceOracle(signer_address, args.nonce)
    else:
        nonce_oracle = RPCNonceOracle(signer_address, conn)

    gas_oracle = None
    if args.gas_price !=None:
        gas_oracle = OverrideGasOracle(price=args.gas_price, conn=conn, code_callback=MinterFaucet.gas)
    else:
        gas_oracle = RPCGasOracle(conn, code_callback=MinterFaucet.gas)

    chain_spec = ChainSpec.from_chain_str(config.get('CIC_CHAIN_SPEC'))

    txf = MinterFaucet(chain_spec, signer=signer, gas_oracle=gas_oracle, nonce_oracle=nonce_oracle)
    (tx_hash_hex, o) = txf.give_to(config.get('_CONTRACT_ADDRESS'), signer_address, config.get('_RECIPIENT'))
    if config.get('_DUMMY'):
        print(tx_hash_hex)
        print(o)
    else:
        r = conn.do(o)
        if block_last:
            r = rpc.wait(tx_hash_hex)
            if r['status'] == 0:
                sys.stderr.write('EVM revert. Wish I had more to tell you')
                sys.exit(1)

        logg.info('faucet gift to {} tx {}'.format(config.get('_RECIPIENT'), tx_hash_hex))

        print(tx_hash_hex)



if __name__ == '__main__':
    main()
